<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: ../index.php');
    exit;
}

require_once '../config/database.php';

// Buscar todos os clientes para o filtro
$stmt = $pdo->prepare("SELECT id, name FROM clients WHERE user_id = ? ORDER BY name");
$stmt->execute([$_SESSION['user_id']]);
$clients = $stmt->fetchAll();

// Filtros
$client_id = filter_input(INPUT_GET, 'client_id', FILTER_SANITIZE_NUMBER_INT);
$status = filter_input(INPUT_GET, 'status', FILTER_SANITIZE_STRING);

// Atualizar status das parcelas vencidas
$pdo->query("
    UPDATE installments 
    SET status = 'overdue' 
    WHERE status = 'pending' 
    AND due_date < CURDATE()
");

// Construir a query base com os filtros
$query = "
    SELECT 
        i.id,
        i.installment_number,
        i.amount,
        i.due_date,
        i.status,
        i.payment_date,
        m.description,
        c.name as client_name,
        m.id as movement_id
    FROM installments i
    JOIN movements m ON i.movement_id = m.id
    JOIN clients c ON m.client_id = c.id
    WHERE m.user_id = ?
";

$params = [$_SESSION['user_id']];

if ($client_id) {
    $query .= " AND m.client_id = ?";
    $params[] = $client_id;
}

if ($status) {
    $query .= " AND i.status = ?";
    $params[] = $status;
}

$query .= " ORDER BY 
    CASE i.status 
        WHEN 'overdue' THEN 1
        WHEN 'pending' THEN 2
        WHEN 'paid' THEN 3
    END,
    i.due_date ASC";

// Executar a query com os filtros
$stmt = $pdo->prepare($query);
$stmt->execute($params);
$receivables = $stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = "Recebíveis";
require_once '../includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php require_once '../includes/sidebar.php'; ?>
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Recebíveis</h1>
            </div>

            <!-- Filtros -->
            <div class="card mb-4">
                <div class="card-body">
                    <form method="GET" class="row g-3">
                        <div class="col-md-4">
                            <label for="client_id" class="form-label">Cliente</label>
                            <select class="form-select" id="client_id" name="client_id">
                                <option value="">Todos os clientes</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>" <?= $client_id == $client['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($client['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label for="status" class="form-label">Status</label>
                            <select class="form-select" id="status" name="status">
                                <option value="">Todos os status</option>
                                <option value="pending" <?= $status == 'pending' ? 'selected' : '' ?>>Pendente</option>
                                <option value="paid" <?= $status == 'paid' ? 'selected' : '' ?>>Pago</option>
                                <option value="overdue" <?= $status == 'overdue' ? 'selected' : '' ?>>Vencido</option>
                            </select>
                        </div>
                        <div class="col-md-4 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary me-2">Filtrar</button>
                            <a href="index.php" class="btn btn-secondary">Limpar</a>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Tabela de Recebíveis -->
            <div class="table-responsive">
                <table class="table table-striped table-hover">
                    <thead>
                        <tr>
                            <th>Cliente</th>
                            <th>Descrição</th>
                            <th>Parcela</th>
                            <th>Valor</th>
                            <th>Vencimento</th>
                            <th>Status</th>
                            <th>Data Pagamento</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($receivables as $receivable): ?>
                            <tr>
                                <td><?= htmlspecialchars($receivable['client_name']) ?></td>
                                <td><?= htmlspecialchars($receivable['description']) ?></td>
                                <td><?= $receivable['installment_number'] ?></td>
                                <td>R$ <?= number_format($receivable['amount'], 2, ',', '.') ?></td>
                                <td><?= date('d/m/Y', strtotime($receivable['due_date'])) ?></td>
                                <td>
                                    <?php
                                    $status_class = '';
                                    $status_text = '';
                                    switch ($receivable['status']) {
                                        case 'paid':
                                            $status_class = 'success';
                                            $status_text = 'Pago';
                                            break;
                                        case 'pending':
                                            $status_class = 'warning';
                                            $status_text = 'Pendente';
                                            break;
                                        case 'overdue':
                                            $status_class = 'danger';
                                            $status_text = 'Vencido';
                                            break;
                                    }
                                    ?>
                                    <span class="badge bg-<?= $status_class ?>"><?= $status_text ?></span>
                                </td>
                                <td>
                                    <?= $receivable['payment_date'] ? date('d/m/Y', strtotime($receivable['payment_date'])) : '-' ?>
                                </td>
                                <td>
                                    <?php if ($receivable['status'] !== 'paid'): ?>
                                        <button class="btn btn-sm btn-success receive-payment" 
                                                data-id="<?= $receivable['id'] ?>"
                                                data-description="<?= htmlspecialchars($receivable['description']) ?>"
                                                data-installment="<?= $receivable['installment_number'] ?>"
                                                data-amount="R$ <?= number_format($receivable['amount'], 2, ',', '.') ?>">
                                            <i class="fas fa-check"></i> Receber
                                        </button>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
</div>

<!-- Modal de Confirmação de Recebimento -->
<div class="modal fade" id="confirmReceiveModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Recebimento</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p><strong>Descrição:</strong> <span id="modalDescription"></span></p>
                <p><strong>Parcela:</strong> <span id="modalInstallment"></span></p>
                <p><strong>Valor:</strong> <span id="modalAmount"></span></p>
                <div class="mb-3">
                    <label for="paymentDate" class="form-label">Data do Recebimento</label>
                    <input type="date" class="form-control" id="paymentDate" required>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <button type="button" class="btn btn-success" id="confirmReceive">Confirmar Recebimento</button>
            </div>
        </div>
    </div>
</div>

<!-- Toasts de Feedback -->
<div class="toast-container position-fixed bottom-0 end-0 p-3">
    <div id="successToast" class="toast align-items-center text-white bg-success border-0" role="alert" aria-live="assertive" aria-atomic="true">
        <div class="d-flex">
            <div class="toast-body">
                Recebimento processado com sucesso!
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    </div>
    <div id="errorToast" class="toast align-items-center text-white bg-danger border-0" role="alert" aria-live="assertive" aria-atomic="true">
        <div class="d-flex">
            <div class="toast-body" id="errorToastMessage">
                Erro ao processar recebimento.
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let selectedInstallmentId = null;
    const modal = new bootstrap.Modal(document.getElementById('confirmReceiveModal'));
    const successToast = new bootstrap.Toast(document.getElementById('successToast'));
    const errorToast = new bootstrap.Toast(document.getElementById('errorToast'));

    // Botões de recebimento
    document.querySelectorAll('.receive-payment').forEach(button => {
        button.addEventListener('click', function() {
            selectedInstallmentId = this.dataset.id;
            
            // Preencher dados do modal
            document.getElementById('modalDescription').textContent = this.dataset.description;
            document.getElementById('modalInstallment').textContent = this.dataset.installment;
            document.getElementById('modalAmount').textContent = this.dataset.amount;
            
            // Configurar data máxima como hoje e valor padrão
            const today = new Date().toISOString().split('T')[0];
            const paymentDateInput = document.getElementById('paymentDate');
            paymentDateInput.max = today;
            paymentDateInput.value = today;
            
            modal.show();
        });
    });

    // Confirmação do recebimento
    document.getElementById('confirmReceive').addEventListener('click', function() {
        if (selectedInstallmentId) {
            const paymentDate = document.getElementById('paymentDate').value;
            if (!paymentDate) {
                document.getElementById('errorToastMessage').textContent = 'Por favor, informe a data do recebimento';
                errorToast.show();
                return;
            }

            const button = document.querySelector(`button[data-id="${selectedInstallmentId}"]`);
            if (button) button.disabled = true;

            fetch('../movements/receive_payment.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `installment_id=${selectedInstallmentId}&payment_date=${paymentDate}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    successToast.show();
                    setTimeout(() => {
                        // Manter os filtros ao recarregar
                        const urlParams = new URLSearchParams(window.location.search);
                        location.href = 'index.php' + (urlParams.toString() ? '?' + urlParams.toString() : '');
                    }, 1000);
                } else {
                    document.getElementById('errorToastMessage').textContent = data.message || 'Erro ao processar recebimento';
                    errorToast.show();
                    if (button) button.disabled = false;
                }
            })
            .catch(error => {
                console.error('Erro:', error);
                document.getElementById('errorToastMessage').textContent = 'Erro ao processar recebimento';
                errorToast.show();
                if (button) button.disabled = false;
            });
        }
        modal.hide();
    });
});
</script>

<?php require_once '../includes/footer.php'; ?>
