<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

require_once 'config/database.php';

// Verificar a estrutura da tabela movements
$stmt = $pdo->query("DESCRIBE movements");
if (!$stmt) {
    die("Erro: A tabela 'movements' pode não existir");
}
$columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
if (!in_array('user_id', $columns)) {
    die("Erro: A coluna 'user_id' não existe na tabela 'movements'");
}

// Primeiro, vamos verificar se conseguimos selecionar as parcelas
$stmt = $pdo->prepare("
    SELECT i.id, i.status, i.due_date
    FROM installments i
    INNER JOIN movements m ON i.movement_id = m.id
    WHERE m.user_id = ?
    LIMIT 1
");

if (!$stmt->execute([$_SESSION['user_id']])) {
    die("Erro na consulta: " . print_r($stmt->errorInfo(), true));
}

// Se chegou aqui, a consulta funcionou, então vamos tentar a atualização
$stmt = $pdo->prepare("
    UPDATE installments i
    INNER JOIN movements m ON i.movement_id = m.id
    SET i.status = 'overdue'
    WHERE i.status = 'pending'
    AND i.due_date < CURDATE()
    AND m.user_id = ?
");

if (!$stmt->execute([$_SESSION['user_id']])) {
    die("Erro na atualização: " . print_r($stmt->errorInfo(), true));
}

// Buscar totais para os cards
$stmt = $pdo->prepare("
    SELECT 
        SUM(CASE WHEN i.status IN ('pending', 'overdue') THEN i.amount ELSE 0 END) as total_receivable,
        SUM(CASE WHEN i.status = 'paid' THEN i.amount ELSE 0 END) as total_received,
        SUM(CASE WHEN i.status = 'overdue' THEN i.amount ELSE 0 END) as total_overdue
    FROM installments i
    INNER JOIN movements m ON i.movement_id = m.id
    WHERE m.user_id = ?
");
$stmt->execute([$_SESSION['user_id']]);
$totals = $stmt->fetch(PDO::FETCH_ASSOC);

$page_title = "Dashboard";
require_once 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php require_once 'includes/sidebar.php'; ?>
        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Dashboard</h1>
            </div>

            <!-- Cards -->
            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="card text-white bg-primary">
                        <div class="card-body">
                            <h5 class="card-title">Total a Receber</h5>
                            <p class="card-text h3">R$ <?= number_format($totals['total_receivable'] ?? 0, 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-success">
                        <div class="card-body">
                            <h5 class="card-title">Total Recebido</h5>
                            <p class="card-text h3">R$ <?= number_format($totals['total_received'] ?? 0, 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="card text-white bg-danger">
                        <div class="card-body">
                            <h5 class="card-title">Total Vencido</h5>
                            <p class="card-text h3">R$ <?= number_format($totals['total_overdue'] ?? 0, 2, ',', '.') ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Gráficos -->
            <div class="row mb-4">
                <!-- Gráfico de Recebimentos por Mês -->
                <div class="col-md-8">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Recebimentos por Mês</h5>
                            <div style="height: 400px;">
                                <canvas id="monthlyChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
                <!-- Gráfico de Status das Parcelas -->
                <div class="col-md-4">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Status das Parcelas</h5>
                            <div style="height: 400px;">
                                <canvas id="statusChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Top 5 Clientes -->
            <div class="row">
                <div class="col-md-12">
                    <div class="card">
                        <div class="card-body">
                            <h5 class="card-title">Top 5 Clientes com Valores Pendentes</h5>
                            <div style="height: 300px;">
                                <canvas id="clientsChart"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>
</div>

<!-- Chart.js -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<script>
// Função para formatar valores em Reais
function formatCurrency(value) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(value);
}

// Função para carregar os dados dos gráficos
function loadChartData() {
    fetch('api/dashboard_charts.php')
        .then(response => response.json())
        .then(data => {
            // Gráfico de Recebimentos por Mês
            const monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
            new Chart(monthlyCtx, {
                type: 'bar',
                data: {
                    labels: data.monthly.map(item => {
                        const [year, month] = item.month.split('-');
                        return new Date(year, month - 1).toLocaleDateString('pt-BR', { month: 'short', year: 'numeric' });
                    }),
                    datasets: [
                        {
                            label: 'A Receber',
                            data: data.monthly.map(item => parseFloat(item.pending_amount) || 0),
                            backgroundColor: '#0d6efd',
                            stack: 'Stack 0'
                        },
                        {
                            label: 'Vencido',
                            data: data.monthly.map(item => parseFloat(item.overdue_amount) || 0),
                            backgroundColor: '#dc3545',
                            stack: 'Stack 0'
                        },
                        {
                            label: 'Recebido',
                            data: data.monthly.map(item => parseFloat(item.paid_amount) || 0),
                            backgroundColor: '#198754',
                            stack: 'Stack 0'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        tooltip: {
                            mode: 'index',
                            intersect: false,
                            callbacks: {
                                label: function(context) {
                                    return context.dataset.label + ': ' + formatCurrency(context.raw);
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            stacked: true
                        },
                        y: {
                            stacked: true,
                            ticks: {
                                callback: function(value) {
                                    return formatCurrency(value);
                                }
                            }
                        }
                    }
                }
            });

            // Gráfico de Status das Parcelas
            const statusCtx = document.getElementById('statusChart').getContext('2d');
            new Chart(statusCtx, {
                type: 'doughnut',
                data: {
                    labels: ['A Receber', 'Recebido', 'Vencido'],
                    datasets: [{
                        data: [
                            parseFloat(data.status_summary.find(item => item.status === 'pending')?.total_amount) || 0,
                            parseFloat(data.status_summary.find(item => item.status === 'paid')?.total_amount) || 0,
                            parseFloat(data.status_summary.find(item => item.status === 'overdue')?.total_amount) || 0
                        ],
                        backgroundColor: ['#0d6efd', '#198754', '#dc3545']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = total > 0 ? ((context.raw / total) * 100).toFixed(1) + '%' : '0%';
                                    return `${context.label}: ${formatCurrency(context.raw)} (${percentage})`;
                                }
                            }
                        }
                    }
                }
            });

            // Gráfico de Top 5 Clientes
            const clientsCtx = document.getElementById('clientsChart').getContext('2d');
            new Chart(clientsCtx, {
                type: 'bar',
                data: {
                    labels: data.top_clients.map(item => item.client_name),
                    datasets: [{
                        label: 'Valor Pendente',
                        data: data.top_clients.map(item => parseFloat(item.pending_amount) || 0),
                        backgroundColor: '#0d6efd'
                    }]
                },
                options: {
                    indexAxis: 'y',
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    return formatCurrency(context.raw);
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            ticks: {
                                callback: function(value) {
                                    return formatCurrency(value);
                                }
                            }
                        }
                    }
                }
            });
        });
}

// Carregar os gráficos quando a página carregar
document.addEventListener('DOMContentLoaded', loadChartData);
</script>

<?php require_once 'includes/footer.php'; ?>
