/**
 * modern-theme.js - Configurações do tema moderno para o dashboard
 * 
 * Este arquivo contém funções e configurações para o tema moderno,
 * incluindo ajustes de cores, animações e outros elementos visuais.
 */

// Configurações globais do tema
const themeConfig = {
    // Cores primárias do tema
    colors: {
        primary: '#4361ee',
        secondary: '#805dca',
        success: '#2dce89',
        info: '#11cdef',
        warning: '#fb6340',
        danger: '#f5365c',
        light: '#eef2f7',
        dark: '#2a2a2a'
    },
    
    // Configurações de animação
    animations: {
        enabled: true,
        speed: 300,
        easing: 'ease'
    },
    
    // Preferências de fonte
    fonts: {
        base: "'Poppins', sans-serif",
        headings: "'Poppins', sans-serif",
        size: '14px'
    }
};

// Aplica as configurações de cores do tema aos elementos CSS
function applyThemeColors() {
    const rootStyle = document.documentElement.style;
    
    // Definir variáveis CSS para as cores do tema
    Object.keys(themeConfig.colors).forEach(colorName => {
        rootStyle.setProperty(`--${colorName}-color`, themeConfig.colors[colorName]);
    });
}

// Configura as animações do dashboard
function setupAnimations() {
    if (!themeConfig.animations.enabled) return;
    
    // Adicionar classe para animações ativadas
    document.body.classList.add('theme-animations-enabled');
    
    // Encontrar todos os elementos com a classe animate-on-scroll
    const animatedElements = document.querySelectorAll('.animate-on-scroll');
    
    // Configurar o observador de interseção para animar elementos quando visíveis
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animated');
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1 });
    
    // Observar todos os elementos animados
    animatedElements.forEach(element => {
        observer.observe(element);
    });
}

// Inicializa o tema
function initModernTheme() {
    applyThemeColors();
    
    // Aguardar o carregamento completo da página para aplicar animações
    window.addEventListener('load', () => {
        setupAnimations();
    });
}

// Exportar funções para uso global
window.modernTheme = {
    config: themeConfig,
    init: initModernTheme,
    applyColors: applyThemeColors,
    setupAnimations: setupAnimations,
    updateDashboardData: function(range) {
        // Redirecionar para a home com o período selecionado
        window.location.href = '/home?periodo=' + range;
    }
};

// Inicializar automaticamente quando o documento estiver pronto
document.addEventListener('DOMContentLoaded', initModernTheme); 