<?php

namespace App\Listeners;

use App\Events\LimiteUsoAtingido;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\LimiteUsoNotificacao;
use App\Services\NotificacaoService;

class NotificarLimiteUso implements ShouldQueue
{
    use InteractsWithQueue;

    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param  LimiteUsoAtingido  $event
     * @return void
     */
    public function handle(LimiteUsoAtingido $event)
    {
        $empresa = $event->empresa;
        
        // Obtém a lista de destinatários
        $destinatarios = NotificacaoService::getDestinatarios();
        
        // Log de início do processo
        Log::info('Iniciando envio de notificação de limite de uso atingido', [
            'empresa_id' => $empresa->id,
            'empresa_nome' => $empresa->nome,
            'tipo_limite' => $event->tipoLimite,
            'valor_atual' => $event->valorAtual,
            'valor_limite' => $event->valorLimite,
            'destinatarios' => $destinatarios
        ]);
        
        try {
            // Envio de e-mail
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new LimiteUsoNotificacao($empresa, $event->tipoLimite, $event->valorAtual, $event->valorLimite));
                Log::info("Email de limite de uso enviado com sucesso para {$dest}");
            }
        } catch (\Exception $e) {
            Log::error('Erro ao enviar email de notificação de limite de uso', [
                'erro' => $e->getMessage(),
                'empresa_id' => $empresa->id,
                'tipo_limite' => $event->tipoLimite
            ]);
        }
    }
} 