<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\NPSPesquisa;
use App\Models\Empresa;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\NPSConfiguracao;

class NPSController extends Controller
{
    /**
     * Construtor
     */
    public function __construct()
    {
        // Middleware para verificar acesso ao painel administrativo (somente SuperAdmin)
        $this->middleware(function ($request, $next) {
            // Verificar se é uma rota de admin
            $rota = $request->route()->getName();
            if (in_array($rota, ['nps.admin', 'nps.detalhes']) && !__isMaster()) {
                return redirect()->route('home')->with('error', 'Acesso negado. Apenas administradores podem acessar esta funcionalidade.');
            }
            return $next($request);
        });
    }
    
    /**
     * Verifica se o usuário deve ver a pesquisa NPS hoje
     * 
     * @return \Illuminate\Http\JsonResponse
     */
    public function verificarExibicao()
    {
        $user = Auth::user();
        
        // Não exibir para SuperAdmin
        if (__isMaster()) {
            return response()->json(['exibir' => false]);
        }
        
        // Verificar se a pesquisa está ativa de acordo com as configurações
        if (!NPSConfiguracao::deveExibirHoje($user->id)) {
            return response()->json(['exibir' => false]);
        }
        
        // Se chegou até aqui, deve exibir a pesquisa
        $configuracao = NPSConfiguracao::obterOuCriar();
        
        return response()->json([
            'exibir' => true,
            'titulo' => $configuracao->titulo,
            'descricao' => $configuracao->descricao
        ]);
    }
    
    /**
     * Salva a resposta da pesquisa NPS
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function salvarResposta(Request $request)
    {
        $user = Auth::user();
        $empresaId = $user->empresa->empresa_id ?? null;
        
        // Validação básica
        $request->validate([
            'nota' => 'nullable|integer|min:0|max:10',
            'feedback' => 'nullable|string|max:1000',
            'ignorar' => 'nullable|boolean'
        ]);
        
        if (!$empresaId) {
            return response()->json(['error' => 'Empresa não encontrada'], 422);
        }
        
        // Salvar resposta na pesquisa
        $dados = [
            'ignorado' => (bool) $request->ignorar
        ];
        
        // Se não for para ignorar, salvamos a nota e feedback
        if (!$request->ignorar) {
            $dados['nota'] = $request->nota;
            $dados['feedback'] = $request->feedback;
        }
        
        $pesquisa = NPSPesquisa::criarOuAtualizarPesquisa($user->id, $empresaId, $dados);
        
        return response()->json(['success' => true, 'message' => 'Resposta salva com sucesso!']);
    }
    
    /**
     * Exibe o painel administrativo com estatísticas do NPS
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function admin(Request $request)
    {
        // Filtros
        $mes = $request->get('mes', Carbon::now()->format('Y-m'));
        $empresaId = $request->get('empresa_id');
        
        // Converter mês para data inicial e final
        $dataInicio = Carbon::createFromFormat('Y-m', $mes)->startOfMonth();
        $dataFim = Carbon::createFromFormat('Y-m', $mes)->endOfMonth();
        
        // Base query
        $query = NPSPesquisa::query()
            ->whereBetween('mes_referencia', [$dataInicio->format('Y-m-d'), $dataFim->format('Y-m-d')]);
            
        // Filtrar por empresa se necessário
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        // Obter resultados
        $pesquisas = $query->with(['usuario', 'empresa'])->get();
        
        // Estatísticas gerais
        $totalRespondentes = $pesquisas->where('respondido', true)->count();
        $totalIgnorados = $pesquisas->where('ignorado', true)->count();
        
        // Calcular o NPS geral
        $detratores = $pesquisas->where('respondido', true)->where('nota', '<=', 6)->count();
        $neutros = $pesquisas->where('respondido', true)->whereBetween('nota', [7, 8])->count();
        $promotores = $pesquisas->where('respondido', true)->whereBetween('nota', [9, 10])->count();
        
        $npsScore = 0;
        if ($totalRespondentes > 0) {
            $npsScore = round((($promotores / $totalRespondentes) - ($detratores / $totalRespondentes)) * 100);
        }
        
        // Estatísticas por empresa
        $estatisticasPorEmpresa = $pesquisas->where('respondido', true)
            ->groupBy('empresa_id')
            ->map(function ($grupo) {
                $totalEmpresa = $grupo->count();
                $detratoresEmpresa = $grupo->where('nota', '<=', 6)->count();
                $promotoresEmpresa = $grupo->whereBetween('nota', [9, 10])->count();
                
                $npsEmpresa = 0;
                if ($totalEmpresa > 0) {
                    $npsEmpresa = round((($promotoresEmpresa / $totalEmpresa) - ($detratoresEmpresa / $totalEmpresa)) * 100);
                }
                
                $mediaNotas = $grupo->avg('nota') ?? 0;
                
                return [
                    'empresa_id' => $grupo->first()->empresa_id,
                    'nome_empresa' => $grupo->first()->empresa->nome ?? 'Desconhecida',
                    'total_respondentes' => $totalEmpresa,
                    'detratores' => $detratoresEmpresa,
                    'neutros' => $grupo->whereBetween('nota', [7, 8])->count(),
                    'promotores' => $promotoresEmpresa,
                    'nps_score' => $npsEmpresa,
                    'media_notas' => round($mediaNotas, 1)
                ];
            })
            ->sortByDesc('nps_score')
            ->values();
            
        // Obter informações de usuários e empresas que ignoraram a pesquisa
        $ignoradosQuery = NPSPesquisa::query()
            ->whereBetween('mes_referencia', [$dataInicio->format('Y-m-d'), $dataFim->format('Y-m-d')])
            ->where('ignorado', true);
            
        // Filtrar por empresa se necessário
        if ($empresaId) {
            $ignoradosQuery->where('empresa_id', $empresaId);
        }
        
        // Obter pesquisas ignoradas com relacionamentos
        $pesquisasIgnoradas = $ignoradosQuery->with(['usuario', 'empresa'])->get();
        
        // Estatísticas de ignorados por empresa
        $ignoradosPorEmpresa = $pesquisasIgnoradas
            ->groupBy('empresa_id')
            ->map(function ($grupo) {
                return [
                    'empresa_id' => $grupo->first()->empresa_id,
                    'nome_empresa' => $grupo->first()->empresa->nome ?? 'Desconhecida',
                    'total_ignorados' => $grupo->count(),
                    'percentual' => 0 // Será calculado abaixo
                ];
            })
            ->values();
            
        // Calcular percentuais de ignorados por empresa
        $totalGeral = $totalRespondentes + $totalIgnorados;
        if ($totalGeral > 0) {
            $ignoradosPorEmpresa = $ignoradosPorEmpresa->map(function ($item) use ($totalGeral) {
                $item['percentual'] = round(($item['total_ignorados'] / $totalGeral) * 100, 1);
                return $item;
            });
        }
        
        // Lista de meses para o filtro (últimos 12 meses)
        $mesesDisponiveis = [];
        for ($i = 0; $i < 12; $i++) {
            $mesData = Carbon::now()->subMonths($i);
            $mesesDisponiveis[$mesData->format('Y-m')] = $mesData->format('F Y');
        }
        
        // Lista de empresas para o filtro
        $empresas = Empresa::orderBy('nome')->get();
        
        return view('nps.admin', compact(
            'pesquisas', 
            'totalRespondentes', 
            'totalIgnorados',
            'detratores',
            'neutros',
            'promotores',
            'npsScore',
            'estatisticasPorEmpresa',
            'pesquisasIgnoradas',
            'ignoradosPorEmpresa',
            'mes',
            'empresaId',
            'mesesDisponiveis',
            'empresas'
        ));
    }
    
    /**
     * Exibe detalhes das respostas com comentários
     * 
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\View\View
     */
    public function detalhes(Request $request)
    {
        // Filtros
        $mes = $request->get('mes', Carbon::now()->format('Y-m'));
        $empresaId = $request->get('empresa_id');
        $tipoResposta = $request->get('tipo', 'todos'); // 'todos', 'detratores', 'neutros', 'promotores'
        
        // Converter mês para data inicial e final
        $dataInicio = Carbon::createFromFormat('Y-m', $mes)->startOfMonth();
        $dataFim = Carbon::createFromFormat('Y-m', $mes)->endOfMonth();
        
        // Base query
        $query = NPSPesquisa::query()
            ->whereBetween('mes_referencia', [$dataInicio->format('Y-m-d'), $dataFim->format('Y-m-d')])
            ->where('respondido', true);
            
        // Filtrar por empresa se necessário
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        // Filtrar por tipo de resposta
        if ($tipoResposta == 'detratores') {
            $query->where('nota', '<=', 6);
        } elseif ($tipoResposta == 'neutros') {
            $query->whereBetween('nota', [7, 8]);
        } elseif ($tipoResposta == 'promotores') {
            $query->whereBetween('nota', [9, 10]);
        }
        
        // Obter resultados
        $respostas = $query->with(['usuario', 'empresa'])->get();
        
        // Preparar lista de meses e empresas para filtros
        $mesesDisponiveis = [];
        for ($i = 0; $i < 12; $i++) {
            $mesData = Carbon::now()->subMonths($i);
            $mesesDisponiveis[$mesData->format('Y-m')] = $mesData->format('F Y');
        }
        
        $empresas = Empresa::orderBy('nome')->get();
        
        return view('nps.detalhes', compact(
            'respostas',
            'mes',
            'empresaId',
            'tipoResposta',
            'mesesDisponiveis',
            'empresas'
        ));
    }
}
