<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

require_once 'config/database.php';
$database = new Database();
$db = $database->getConnection();

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'criar' || $_POST['action'] === 'editar') {
        $usuario_id = $_SESSION['user_id'];
        $descricao = $_POST['descricao'];
        
        // Corrigir a conversão do valor
        $valor = $_POST['valor'];
        $valor = str_replace('.', '', $valor); // Remove os pontos de milhar
        $valor = str_replace(',', '.', $valor); // Converte vírgula em ponto
        
        $tipo = $_POST['tipo'];
        $forma_pagamento_id = $_POST['forma_pagamento_id'];
        $data_lancamento = $_POST['data_lancamento'];

        // Verificar se é Vale e se é uma entrada
        $query_forma = "SELECT nome FROM formas_pagamento WHERE id = :id";
        $stmt_forma = $db->prepare($query_forma);
        $stmt_forma->bindParam(':id', $forma_pagamento_id);
        $stmt_forma->execute();
        $forma_nome = $stmt_forma->fetchColumn();

        if ($forma_nome === 'Vale' && $tipo === 'entrada') {
            $error = "Não é permitido fazer lançamentos de entrada com Vale.";
        } else {
            // Define entrada ou saída baseado no tipo
            $entrada = $tipo === 'entrada' ? $valor : 0;
            $saida = $tipo === 'saida' ? $valor : 0;
            
            if ($_POST['action'] === 'criar') {
                $query = "INSERT INTO lancamentos (usuario_id, descricao, valor, entrada, saida, tipo, forma_pagamento_id, data_lancamento) 
                         VALUES (:usuario_id, :descricao, :valor, :entrada, :saida, :tipo, :forma_pagamento_id, :data_lancamento)";
            } else {
                $id = $_POST['id'];
                $query = "UPDATE lancamentos 
                         SET descricao = :descricao, 
                             valor = :valor,
                             entrada = :entrada,
                             saida = :saida,
                             tipo = :tipo,
                             forma_pagamento_id = :forma_pagamento_id,
                             data_lancamento = :data_lancamento
                         WHERE id = :id AND usuario_id = :usuario_id";
            }
            
            $stmt = $db->prepare($query);
            $stmt->bindParam(':usuario_id', $usuario_id);
            $stmt->bindParam(':descricao', $descricao);
            $stmt->bindParam(':valor', $valor);
            $stmt->bindParam(':entrada', $entrada);
            $stmt->bindParam(':saida', $saida);
            $stmt->bindParam(':tipo', $tipo);
            $stmt->bindParam(':forma_pagamento_id', $forma_pagamento_id);
            $stmt->bindParam(':data_lancamento', $data_lancamento);
            
            if ($_POST['action'] === 'editar') {
                $stmt->bindParam(':id', $id);
            }
            
            if ($stmt->execute()) {
                // Se for Promissória e for uma entrada, criar conta a receber
                if ($forma_nome === 'Promissória' && $tipo === 'entrada') {
                    $lancamento_id = $_POST['action'] === 'criar' ? $db->lastInsertId() : $id;
                    $data_vencimento = date('Y-m-d', strtotime($data_lancamento . ' + 30 days'));
                    
                    $query_conta = "INSERT INTO contas_receber (lancamento_id, usuario_id, valor, data_vencimento, status) 
                                  VALUES (:lancamento_id, :usuario_id, :valor, :data_vencimento, 'PENDENTE')";
                    $stmt_conta = $db->prepare($query_conta);
                    $stmt_conta->bindParam(':lancamento_id', $lancamento_id);
                    $stmt_conta->bindParam(':usuario_id', $usuario_id);
                    $stmt_conta->bindParam(':valor', $valor);
                    $stmt_conta->bindParam(':data_vencimento', $data_vencimento);
                    $stmt_conta->execute();
                }
                
                header("Location: lancamentos.php");
                exit();
            }
        }
    } elseif ($_POST['action'] === 'excluir' && isset($_POST['id'])) {
        $id = $_POST['id'];
        $justificativa = $_POST['justificativa'];
        
        // Iniciar transação
        $db->beginTransaction();
        
        try {
            // Buscar dados do lançamento antes de excluir
            $query_dados = "SELECT * FROM lancamentos WHERE id = :id AND usuario_id = :usuario_id";
            $stmt_dados = $db->prepare($query_dados);
            $stmt_dados->bindParam(':id', $id);
            $stmt_dados->bindParam(':usuario_id', $_SESSION['user_id']);
            $stmt_dados->execute();
            $dados_anteriores = $stmt_dados->fetch(PDO::FETCH_ASSOC);
            
            if ($dados_anteriores) {
                // Primeiro, verificar e excluir registros relacionados em contas_receber
                $query_contas = "SELECT * FROM contas_receber WHERE lancamento_id = :lancamento_id";
                $stmt_contas = $db->prepare($query_contas);
                $stmt_contas->bindParam(':lancamento_id', $id);
                $stmt_contas->execute();
                $conta_receber = $stmt_contas->fetch(PDO::FETCH_ASSOC);
                
                if ($conta_receber) {
                    // Se encontrou conta a receber, registrar log dela primeiro
                    $query_log_conta = "INSERT INTO logs (usuario_id, acao, tabela, registro_id, dados_anteriores, justificativa, ip) 
                                      VALUES (:usuario_id, 'excluir', 'contas_receber', :registro_id, :dados_anteriores, :justificativa, :ip)";
                    $stmt_log_conta = $db->prepare($query_log_conta);
                    $stmt_log_conta->bindParam(':usuario_id', $_SESSION['user_id']);
                    $stmt_log_conta->bindParam(':registro_id', $conta_receber['id']);
                    $stmt_log_conta->bindParam(':dados_anteriores', json_encode($conta_receber));
                    $stmt_log_conta->bindParam(':justificativa', $justificativa);
                    $stmt_log_conta->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
                    $stmt_log_conta->execute();
                    
                    // Excluir a conta a receber
                    $query_del_conta = "DELETE FROM contas_receber WHERE id = :id";
                    $stmt_del_conta = $db->prepare($query_del_conta);
                    $stmt_del_conta->bindParam(':id', $conta_receber['id']);
                    $stmt_del_conta->execute();
                }
                
                // Registrar log do lançamento
                $query_log = "INSERT INTO logs (usuario_id, acao, tabela, registro_id, dados_anteriores, justificativa, ip) 
                             VALUES (:usuario_id, 'excluir', 'lancamentos', :registro_id, :dados_anteriores, :justificativa, :ip)";
                $stmt_log = $db->prepare($query_log);
                $stmt_log->bindParam(':usuario_id', $_SESSION['user_id']);
                $stmt_log->bindParam(':registro_id', $id);
                $stmt_log->bindParam(':dados_anteriores', json_encode($dados_anteriores));
                $stmt_log->bindParam(':justificativa', $justificativa);
                $stmt_log->bindParam(':ip', $_SERVER['REMOTE_ADDR']);
                $stmt_log->execute();
                
                // Excluir o lançamento
                $query = "DELETE FROM lancamentos WHERE id = :id AND usuario_id = :usuario_id";
                $stmt = $db->prepare($query);
                $stmt->bindParam(':id', $id);
                $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
                $stmt->execute();
                
                // Commit da transação
                $db->commit();
                
                header("Location: lancamentos.php");
                exit();
            }
        } catch (Exception $e) {
            // Rollback em caso de erro
            $db->rollBack();
            $error = "Erro ao excluir lançamento: " . $e->getMessage();
        }
    }
}

// Buscar formas de pagamento ativas
$query = "SELECT id, nome FROM formas_pagamento WHERE ativo = 1 ORDER BY nome";
$stmt = $db->prepare($query);
$stmt->execute();
$formas_pagamento = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Processar filtros
$data_inicial = isset($_GET['data_inicial']) ? $_GET['data_inicial'] : date('Y-m-01'); // Primeiro dia do mês
$data_final = isset($_GET['data_final']) ? $_GET['data_final'] : date('Y-m-t'); // Último dia do mês

// Buscar lançamentos
$query = "SELECT l.*, f.nome as forma_pagamento, 
                 DATE_FORMAT(l.data_lancamento, '%d/%m/%Y') as data_formatada,
                 DATE_FORMAT(l.data_lancamento, '%Y-%m-%d') as data_iso 
          FROM lancamentos l 
          JOIN formas_pagamento f ON l.forma_pagamento_id = f.id 
          WHERE 1=1";

// Verificar se o usuário é admin
$query_admin = "SELECT admin FROM usuarios WHERE id = :user_id";
$stmt_admin = $db->prepare($query_admin);
$stmt_admin->bindParam(':user_id', $_SESSION['user_id']);
$stmt_admin->execute();
$is_admin = $stmt_admin->fetchColumn();

// Se não for admin, filtrar apenas os lançamentos do usuário
if (!$is_admin) {
    $query .= " AND l.usuario_id = :usuario_id";
}

if ($data_inicial) {
    $query .= " AND l.data_lancamento >= :data_inicial";
}

if ($data_final) {
    $query .= " AND l.data_lancamento <= :data_final";
}

$query .= " ORDER BY l.data_lancamento DESC, l.id DESC";

$stmt = $db->prepare($query);

// Bind parameter apenas se não for admin
if (!$is_admin) {
    $stmt->bindParam(':usuario_id', $_SESSION['user_id']);
}

if ($data_inicial) {
    $stmt->bindParam(':data_inicial', $data_inicial);
}

if ($data_final) {
    $stmt->bindParam(':data_final', $data_final);
}

$stmt->execute();
$lancamentos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calcular totais
$total_entradas = 0;
$total_saidas = 0;

// Query para calcular totais
$query_totais = "SELECT 
    SUM(CASE WHEN entrada > 0 THEN entrada ELSE 0 END) as total_entradas,
    SUM(CASE WHEN saida > 0 THEN saida ELSE 0 END) as total_saidas
FROM lancamentos 
WHERE 1=1";

// Se não for admin, filtrar apenas os lançamentos do usuário
if (!$is_admin) {
    $query_totais .= " AND usuario_id = :usuario_id";
}

if ($data_inicial) {
    $query_totais .= " AND data_lancamento >= :data_inicial";
}

if ($data_final) {
    $query_totais .= " AND data_lancamento <= :data_final";
}

$stmt_totais = $db->prepare($query_totais);

// Bind parameters
if (!$is_admin) {
    $stmt_totais->bindParam(':usuario_id', $_SESSION['user_id']);
}

if ($data_inicial) {
    $stmt_totais->bindParam(':data_inicial', $data_inicial);
}

if ($data_final) {
    $stmt_totais->bindParam(':data_final', $data_final);
}

$stmt_totais->execute();
$totais = $stmt_totais->fetch(PDO::FETCH_ASSOC);

$total_entradas = $totais['total_entradas'] ?? 0;
$total_saidas = $totais['total_saidas'] ?? 0;

$saldo = $total_entradas - $total_saidas;

// Definir título da página
$titulo = "Lançamentos";

// Conteúdo da página
ob_start();
?>

<!-- Modal de Lançamento -->
<div class="modal fade" id="lancamentoModal" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Lançamento</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="lancamentoForm" method="POST">
                    <input type="hidden" name="action" value="criar">
                    <input type="hidden" name="id" value="">
                    
                    <div class="mb-3">
                        <label for="descricao" class="form-label">Descrição/Nome</label>
                        <input type="text" class="form-control" id="descricao" name="descricao" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="valor" class="form-label">Valor</label>
                        <input type="text" class="form-control" id="valor" name="valor" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="tipo" class="form-label">Tipo</label>
                        <select class="form-select" id="tipo" name="tipo" required>
                            <option value="entrada">Entrada</option>
                            <option value="saida">Saída</option>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="forma_pagamento_id" class="form-label">Forma de Pagamento</label>
                        <select class="form-select" id="forma_pagamento_id" name="forma_pagamento_id" required>
                            <option value="">Selecione...</option>
                            <?php foreach ($formas_pagamento as $forma): ?>
                                <option value="<?php echo $forma['id']; ?>">
                                    <?php echo htmlspecialchars($forma['nome']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="mb-3">
                        <label for="data_lancamento" class="form-label">Data</label>
                        <input type="date" class="form-control" id="data_lancamento" name="data_lancamento" required>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                <button type="submit" form="lancamentoForm" class="btn btn-primary">Salvar</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal de Exclusão -->
<div class="modal fade" id="modalExclusao" tabindex="-1" data-bs-backdrop="static" data-bs-keyboard="false">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Confirmar Exclusão</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="formExclusao" method="POST">
                    <input type="hidden" name="action" value="excluir">
                    <input type="hidden" name="id" id="exclusao_id">
                    
                    <div class="mb-3">
                        <label for="justificativa" class="form-label">Justificativa da Exclusão</label>
                        <textarea class="form-control" id="justificativa" name="justificativa" rows="3" required></textarea>
                    </div>
                    
                    <div class="text-end">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-danger">Confirmar Exclusão</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Filtros -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Filtros</h5>
    </div>
    <div class="card-body">
        <form method="GET" id="formFiltro" class="row g-3">
            <div class="col-md-4">
                <label for="data_inicial" class="form-label">Data Inicial</label>
                <input type="date" class="form-control" id="data_inicial" name="data_inicial" 
                       value="<?php echo $data_inicial; ?>">
            </div>
            
            <div class="col-md-4">
                <label for="data_final" class="form-label">Data Final</label>
                <input type="date" class="form-control" id="data_final" name="data_final" 
                       value="<?php echo $data_final; ?>">
            </div>
            
            <div class="col-md-4 d-flex align-items-end">
                <div class="btn-group w-100">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-filter"></i> Filtrar
                    </button>
                    <a href="lancamentos.php" class="btn btn-secondary">
                        <i class="fas fa-eraser"></i> Limpar Filtros
                    </a>
                    <a href="relatorio.php?data_inicial=<?php echo $data_inicial; ?>&data_final=<?php echo $data_final; ?>" 
                       class="btn btn-dark" target="_blank">
                        <i class="fas fa-file-alt"></i> Relatório
                    </a>
                </div>
            </div>
        </form>
    </div>
</div>

<!-- Conteúdo Principal -->
<div class="row mb-4">
    <div class="col-md-4">
        <div class="card bg-success text-white">
            <div class="card-body">
                <h5 class="card-title">Total Entradas</h5>
                <h3 class="mb-0">R$ <?php echo number_format($total_entradas, 2, ',', '.'); ?></h3>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card bg-danger text-white">
            <div class="card-body">
                <h5 class="card-title">Total Saídas</h5>
                <h3 class="mb-0">R$ <?php echo number_format($total_saidas, 2, ',', '.'); ?></h3>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card <?php echo $saldo >= 0 ? 'bg-primary' : 'bg-danger'; ?> text-white">
            <div class="card-body">
                <h5 class="card-title">Saldo</h5>
                <h3 class="mb-0">R$ <?php echo number_format($saldo, 2, ',', '.'); ?></h3>
            </div>
        </div>
    </div>
</div>

<!-- Tabela de Lançamentos -->
<div class="card">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Lançamentos</h5>
        <button type="button" class="btn btn-primary" onclick="novoLancamento()">
            Novo Lançamento
        </button>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table" id="lancamentosTable">
                <thead>
                    <tr>
                        <th>Data</th>
                        <th>Descrição</th>
                        <th>Forma de Pagamento</th>
                        <th>Entrada</th>
                        <th>Saída</th>
                        <th>Tipo</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($lancamentos as $lancamento): ?>
                        <tr>
                            <td><?php echo $lancamento['data_formatada']; ?></td>
                            <td><?php echo htmlspecialchars($lancamento['descricao']); ?></td>
                            <td><?php echo htmlspecialchars($lancamento['forma_pagamento']); ?></td>
                            <td class="text-success">
                                R$ <?php echo number_format($lancamento['entrada'] > 0 ? $lancamento['entrada'] : 0, 2, ',', '.'); ?>
                            </td>
                            <td class="text-danger">
                                R$ <?php echo number_format($lancamento['saida'] > 0 ? $lancamento['saida'] : 0, 2, ',', '.'); ?>
                            </td>
                            <td>
                                <span class="badge <?php echo $lancamento['entrada'] > 0 ? 'bg-success' : 'bg-danger'; ?>">
                                    <?php echo $lancamento['entrada'] > 0 ? 'Entrada' : 'Saída'; ?>
                                </span>
                            </td>
                            <td>
                                <div class="btn-group">
                                    <button type="button" class="btn btn-sm btn-primary" onclick='editarLancamento(<?php echo json_encode($lancamento); ?>)'>
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" onclick='excluirLancamento(<?php echo $lancamento["id"]; ?>)'>
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php
$content = ob_get_clean();

// JavaScript adicional
$extra_js = '
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery.mask/1.14.16/jquery.mask.min.js"></script>
<script>
$(document).ready(function() {
    // Inicializar DataTable
    $("#lancamentosTable").DataTable({
        // Configurar o idioma para Português do Brasil
        language: {
            url: "//cdn.datatables.net/plug-ins/1.11.5/i18n/pt-BR.json"
        },
        // Definir a ordem padrão como data decrescente (mais recente primeiro)
        order: [[0, "desc"]],
        // Configurar o número de registros por página
        pageLength: 5,
        // Opções de quantidade de registros por página que o usuário pode escolher
        lengthMenu: [[5, 10, 25, 50], [5, 10, 25, 50]],
        // Configurar os botões e elementos da tabela
        dom: "Bfrtip",
        // Configurar os botões disponíveis
        buttons: [
            {
                extend: "print",
                text: "<i class=\"fas fa-print\"></i> Imprimir",
                className: "btn btn-dark d-none",
                exportOptions: {
                    columns: [0, 1, 2, 3, 4, 5]
                }
            }
        ]
    });
    
    // Máscara para o campo de valor com formatação automática
    $("#valor").mask("#.##0,00", {reverse: true});
    
    // Definir valor inicial como 0,00
    if ($("#valor").val() === "") {
        $("#valor").val("0,00");
    }
    
    // Quando o campo recebe foco, limpa se for 0,00
    $("#valor").focus(function() {
        if ($(this).val() === "0,00") {
            $(this).val("");
        }
    });
    
    // Quando o campo perde foco, coloca 0,00 se estiver vazio
    $("#valor").blur(function() {
        if ($(this).val() === "") {
            $(this).val("0,00");
        }
    });
});

function novoLancamento() {
    $("#lancamentoForm")[0].reset();
    $("#lancamentoForm [name=action]").val("criar");
    $("#lancamentoForm [name=id]").val("");
    
    // Pegar a data atual no fuso horário local
    const hoje = new Date();
    const dataLocal = hoje.getFullYear() + "-" + 
                     String(hoje.getMonth() + 1).padStart(2, "0") + "-" + 
                     String(hoje.getDate()).padStart(2, "0");
    
    $("#data_lancamento").val(dataLocal);
    $("#lancamentoModal").modal("show");
}

function editarLancamento(lancamento) {
    $("#lancamentoForm")[0].reset();
    $("#lancamentoForm [name=action]").val("editar");
    $("#lancamentoForm [name=id]").val(lancamento.id);
    $("#descricao").val(lancamento.descricao);
    $("#valor").val(number_format(lancamento.entrada > 0 ? lancamento.entrada : lancamento.saida, 2, ",", "."));
    $("#tipo").val(lancamento.entrada > 0 ? "entrada" : "saida");
    $("#forma_pagamento_id").val(lancamento.forma_pagamento_id);
    $("#data_lancamento").val(lancamento.data_iso);
    $("#lancamentoModal").modal("show");
}

function excluirLancamento(id) {
    $("#exclusao_id").val(id);
    var modal = new bootstrap.Modal(document.getElementById("modalExclusao"));
    modal.show();
}

function number_format(number, decimals, dec_point, thousands_sep) {
    number = (number + "").replace(/[^0-9+\-Ee.]/g, "");
    var n = !isFinite(+number) ? 0 : +number,
        prec = !isFinite(+decimals) ? 0 : Math.abs(decimals),
        sep = (typeof thousands_sep === "undefined") ? "." : thousands_sep,
        dec = (typeof dec_point === "undefined") ? "," : dec_point,
        s = "",
        toFixedFix = function (n, prec) {
            var k = Math.pow(10, prec);
            return "" + Math.round(n * k) / k;
        };
    s = (prec ? toFixedFix(n, prec) : "" + Math.round(n)).split(".");
    if (s[0].length > 3) {
        s[0] = s[0].replace(/\B(?=(?:\d{3})+(?!\d))/g, sep);
    }
    if ((s[1] || "").length < prec) {
        s[1] = s[1] || "";
        s[1] += new Array(prec - s[1].length + 1).join("0");
    }
    return s.join(dec);
}

// Validar datas do filtro
$("#formFiltro").submit(function(e) {
    var dataInicial = $("#data_inicial").val();
    var dataFinal = $("#data_final").val();
    
    if (dataInicial && dataFinal && dataInicial > dataFinal) {
        e.preventDefault();
        alert("A data inicial não pode ser maior que a data final!");
    }
});
</script>';

// Incluir o layout
require_once 'includes/layout.php';
?>
