// Funções para manipulação de cheques
function calcularJuros() {
    const cliente = $('#cliente').val();
    const numeroCheque = $('#numero_cheque').val();
    const banco = $('#banco').val();
    const valor = $('#valor').val().replace(/\./g, '').replace(',', '.');
    const dataTroca = new Date($('#data_troca').val());
    const dataVencimento = new Date($('#data_vencimento').val());

    if (!cliente || isNaN(valor) || valor <= 0 || !dataTroca || !dataVencimento) {
        alert('Por favor, preencha todos os campos obrigatórios corretamente.');
        return;
    }

    // Calcular diferença de dias
    const diffTime = Math.abs(dataVencimento - dataTroca);
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    // Calcular juros (0.1% ao dia = 0.001)
    const jurosDiario = 0.001; // 0.1% ao dia
    const juros = parseFloat(valor) * jurosDiario * diffDays;
    const valorLiquido = parseFloat(valor) - juros;

    // Armazenar cálculo atual
    calculoAtual = {
        cliente,
        numeroCheque,
        banco,
        valor: parseFloat(valor),
        dataTroca,
        dataVencimento,
        diffDays,
        juros: parseFloat(juros.toFixed(2)),
        valorLiquido: parseFloat(valorLiquido.toFixed(2))
    };

    // Mostrar resultados
    $('#valorChequeResult').text(formatarMoeda(valor));
    $('#jurosResult').text(formatarMoeda(juros));
    $('#valorLiquidoResult').text(formatarMoeda(valorLiquido));
    $('#diasResult').text(diffDays + ' dias');

    // Mostrar resultado e botão de salvar
    $('#resultado').show();
    $('.btn-salvar').show();
}

function salvarCheque() {
    if (!calculoAtual) return;

    if (!confirm('Confirma o salvamento deste cheque?')) {
        return;
    }

    // Preparar dados para envio
    const dados = {
        cliente: calculoAtual.cliente,
        numero_cheque: calculoAtual.numeroCheque,
        banco: calculoAtual.banco,
        valor: calculoAtual.valor,
        data_troca: calculoAtual.dataTroca.toISOString().split('T')[0],
        data_vencimento: calculoAtual.dataVencimento.toISOString().split('T')[0],
        juros: calculoAtual.juros,
        valor_liquido: calculoAtual.valorLiquido
    };

    // Enviar dados para o servidor
    $.ajax({
        url: 'salvar_cheque.php',
        type: 'POST',
        contentType: 'application/json',
        data: JSON.stringify(dados),
        success: function(response) {
            if (response.success) {
                alert('Cheque salvo com sucesso!');
                window.location.reload();
            } else {
                alert('Erro ao salvar o cheque: ' + response.error);
            }
        },
        error: function(xhr, status, error) {
            alert('Erro ao salvar o cheque: ' + error);
        }
    });
}

function excluirCheque(chequeId) {
    if (!chequeId) {
        alert('ID do cheque não fornecido');
        return;
    }

    if (confirm('Tem certeza que deseja excluir este cheque?')) {
        $.ajax({
            url: 'excluir_cheque.php',
            type: 'POST',
            data: { cheque_id: chequeId },
            success: function(response) {
                if (response.success) {
                    alert(response.message);
                    window.location.reload();
                } else {
                    alert('Erro: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro na requisição:', error);
                console.error('Status:', status);
                console.error('Resposta:', xhr.responseText);
                alert('Erro ao tentar excluir o cheque. Verifique o console para mais detalhes.');
            }
        });
    }
}

function quitarCheque(chequeId) {
    if (confirm('Confirma a quitação deste cheque?')) {
        console.log('Enviando requisição para marcar_pago.php com ID:', chequeId);
        $.ajax({
            url: 'marcar_pago.php',
            type: 'POST',
            data: { id: chequeId },
            success: function(response) {
                console.log('Resposta recebida:', response);
                try {
                    const data = typeof response === 'string' ? JSON.parse(response) : response;
                    if (data.success) {
                        alert('Cheque quitado com sucesso!');
                        window.location.reload();
                    } else {
                        alert('Erro ao quitar o cheque: ' + data.message);
                    }
                } catch (e) {
                    console.error('Erro ao processar resposta:', e);
                    console.error('Resposta recebida:', response);
                    alert('Erro ao processar resposta do servidor');
                }
            },
            error: function(xhr, status, error) {
                console.error('Erro na requisição:');
                console.error('Status:', status);
                console.error('Erro:', error);
                console.error('Resposta:', xhr.responseText);
                alert('Erro ao tentar quitar o cheque. Verifique o console para mais detalhes.');
            }
        });
    }
}

function formatarMoeda(valor) {
    return new Intl.NumberFormat('pt-BR', {
        style: 'currency',
        currency: 'BRL'
    }).format(valor);
}

// Inicialização quando o documento estiver pronto
$(document).ready(function() {
    // Máscara para campos de moeda
    $('.money').mask('#.##0,00', {reverse: true});
    
    // Validação de datas
    $('#data_troca, #data_vencimento').change(function() {
        const dataVencimento = new Date($('#data_vencimento').val());
        const dataTroca = new Date($('#data_troca').val());
        
        if (dataVencimento < dataTroca) {
            alert('A data de vencimento não pode ser anterior à data da troca!');
            $(this).val('');
        }
    });
});
