<?php
// Listagem direta da pasta uploads (sem depender do banco)
$uploadsDirFs = __DIR__ . '/uploads';
$videos = [];

if (is_dir($uploadsDirFs)) {
    $files = @scandir($uploadsDirFs);
    if ($files) {
        foreach ($files as $f) {
            if ($f === '.' || $f === '..') continue;
            $ext = strtolower(pathinfo($f, PATHINFO_EXTENSION));
            if (!in_array($ext, ['mp4', 'webm'], true)) continue;

            $fullPath = $uploadsDirFs . '/' . $f;
            $mtime = @filemtime($fullPath);

            $videos[] = [
                'nome_arquivo' => $f,
                'caminho' => 'uploads/' . $f,
                'data_envio' => $mtime ? date('Y-m-d H:i:s', $mtime) : ''
            ];
        }

        // Ordena por data de modificação desc
        usort($videos, function ($a, $b) {
            return strcmp($b['data_envio'], $a['data_envio']);
        });
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Galeria de Vídeos</title>
    <link rel="stylesheet" href="style.css" />
</head>
<body>
    <header class="topbar">
        <div class="container topbar-inner">
            <div class="brand">
                <span class="brand-icon">▶</span>
                <span class="brand-text">Vídeo Slide</span>
            </div>
            <!-- <form class="search" onsubmit="event.preventDefault();">
                <input type="text" id="searchInput" placeholder="Pesquisar vídeos" aria-label="Pesquisar vídeos" />
            </form>
            <div class="actions">
                <a href="#upload" class="btn">Upload</a>
            </div> -->
        </div>
    </header>

    <main class="container">
        <!-- <section id="upload" class="upload-section card">
            <h2>Upload de Vídeo</h2>
            <form id="uploadForm" action="upload.php" method="post" enctype="multipart/form-data">
                <div class="form-row">
                    <input type="file" id="videoFile" name="video" accept="video/mp4,video/webm" required />
                </div>
                <div class="form-row">
                    <button type="submit" class="btn primary">Enviar Vídeo</button>
                </div>
                <div class="progress-wrapper" id="progressWrapper" hidden>
                    <progress id="uploadProgress" value="0" max="100"></progress>
                    <span id="progressText">0%</span>
                </div>
                <div id="message" class="message" aria-live="polite"></div>
            </form>
            <p class="hint">Formatos aceitos: .mp4, .webm</p>
        </section> -->

        <section class="player-section card">
            <h2>Player</h2>
            <video id="player" class="player" controls preload="metadata">
                <source id="playerSource" src="" type="video/mp4" />
                Seu navegador não suporta o elemento de vídeo.
            </video>
        </section>

        <section class="gallery-section card">
            <h2>Vídeos Enviados</h2>
            <?php if (empty($videos)) : ?>
                <p class="empty">Nenhum vídeo enviado ainda.</p>
            <?php else : ?>
                <ul class="video-list">
                    <?php foreach ($videos as $v) : ?>
                        <li class="video-item" role="button" tabindex="0"
                            data-path="<?= htmlspecialchars($v['caminho']) ?>"
                            data-type="<?= pathinfo($v['caminho'], PATHINFO_EXTENSION) ?>"
                            data-name="<?= htmlspecialchars($v['nome_arquivo']) ?>"
                            aria-label="Reproduzir <?= htmlspecialchars($v['nome_arquivo']) ?>"
                            onclick="playCard(this)"
                            onkeydown="if(event.key==='Enter'||event.key===' '){event.preventDefault(); playCard(this);}"
                        >
                            <div class="thumb">
                                <div class="play-overlay">▶</div>
                            </div>
                            <div class="video-meta">
                                <strong class="video-name"><?= htmlspecialchars($v['nome_arquivo']) ?></strong>
                                <small class="video-date">Enviado em: <?= htmlspecialchars($v['data_envio']) ?></small>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </section>
    </main>

    <footer class="footer">
        <div class="container">
            <small>&copy; <?= date('Y') ?> Vídeo Slide</small>
        </div>
    </footer>

    <script>
        const allowedTypes = ['video/mp4', 'video/webm'];

        function playCard(card) {
            const path = card.getAttribute('data-path');
            const typeExt = card.getAttribute('data-type');
            const player = document.getElementById('player');
            const source = document.getElementById('playerSource');

            source.src = path;
            source.type = (typeExt && typeExt.toLowerCase() === 'webm') ? 'video/webm' : 'video/mp4';
            player.load();
            player.play();

            // destaque de seleção
            document.querySelectorAll('.video-item.active').forEach(el => el.classList.remove('active'));
            card.classList.add('active');

            // rolar até o player para foco visual
            player.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }

        const uploadForm = document.getElementById('uploadForm');
        const fileInput = document.getElementById('videoFile');
        const progressWrapper = document.getElementById('progressWrapper');
        const progress = document.getElementById('uploadProgress');
        const progressText = document.getElementById('progressText');
        const message = document.getElementById('message');
        const searchInput = document.getElementById('searchInput');

        if (uploadForm) { uploadForm.addEventListener('submit', function (e) {
            e.preventDefault();
            message.textContent = '';

            const file = fileInput.files[0];
            if (!file) {
                message.textContent = 'Selecione um arquivo de vídeo.';
                return;
            }
            if (!allowedTypes.includes(file.type)) {
                message.textContent = 'Tipo de arquivo inválido. Use .mp4 ou .webm';
                return;
            }
            // Sem limite de tamanho aqui; dependerá da configuração do PHP.

            const formData = new FormData(uploadForm);
            const xhr = new XMLHttpRequest();
            xhr.open('POST', uploadForm.action, true);
            xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');

            xhr.upload.addEventListener('progress', (event) => {
                if (event.lengthComputable) {
                    const percent = Math.round((event.loaded / event.total) * 100);
                    progress.value = percent;
                    progressText.textContent = percent + '%';
                    progressWrapper.hidden = false;
                }
            });

            xhr.onload = function () {
                progressWrapper.hidden = true;
                try {
                    const resp = JSON.parse(xhr.responseText);
                    if (resp.success) {
                        message.classList.remove('error');
                        message.classList.add('success');
                        message.textContent = 'Upload realizado com sucesso!';
                        setTimeout(() => { window.location.reload(); }, 800);
                    } else {
                        message.classList.remove('success');
                        message.classList.add('error');
                        message.textContent = resp.error || 'Falha no upload';
                    }
                } catch (err) {
                    message.classList.add('error');
                    message.textContent = 'Erro inesperado no servidor.';
                }
            };

            xhr.onerror = function () {
                progressWrapper.hidden = true;
                message.classList.add('error');
                message.textContent = 'Erro de rede durante o upload.';
            };

            xhr.send(formData);
        }); }

        // Filtro de busca estilo YouTube (por nome do arquivo)
        if (searchInput) {
            searchInput.addEventListener('input', function () {
                const q = this.value.trim().toLowerCase();
                document.querySelectorAll('.video-item').forEach(item => {
                    const name = (item.getAttribute('data-name') || '').toLowerCase();
                    item.style.display = name.includes(q) ? '' : 'none';
                });
            });
        }
    </script>
</body>
</html>