<?php

namespace App\Filament\User\Resources;

use App\Filament\User\Resources\FormasPagamentoResource\Pages;
use App\Filament\User\Resources\FormasPagamentoResource\RelationManagers;
use App\Models\FormasPagamento;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Section;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Enums\Status;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Notifications\Notification;

class FormasPagamentoResource extends Resource
{
    public static function getModelLabel(): string
    {
        return 'Forma de Pagamento';
    }

    public static function getPluralModelLabel(): string
    {
        return 'Formas de Pagamento';
    }

    public static function getNavigationLabel(): string
    {
        return 'Formas de Pagamento';
    }

    public static function mutateFormDataBeforeCreate(array $data): array
    {
        $data['tenant_id'] = auth()->user()->tenant_id;
        return $data;
    }
    protected static ?string $model = FormasPagamento::class;

    protected static ?string $navigationIcon = 'heroicon-o-credit-card';
    protected static ?string $navigationGroup = 'Cadastros';
    protected static ?int $navigationSort = 5;


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Informações Básicas')
                    ->description('Informações principais da forma de pagamento')
                    ->schema([
                        TextInput::make('nome')
                            ->label('Nome')
                            ->required()
                            ->maxLength(50),

                        Select::make('status')
                            ->options([
                                'ativo' => 'Ativo',
                                'inativo' => 'Inativo',
                            ])
                            ->default('ativo'),

                        Select::make('tenant_id')
                            ->label('Empresa')
                            ->relationship('tenant', 'name', fn ($query) => $query->where('id', auth()->user()->tenant_id))
                            ->required()
                            ->preload()
                            ->searchable()
                            ->default(auth()->user()->tenant_id)
                            ->disabled()
                            ->dehydrated()
                ])->columns(2),

                Section::make('Configurações')
                    ->description('Configure as opções da forma de pagamento')
                    ->schema([
                        Toggle::make('crediario')
                            ->label('É Crediário')
                            ->helperText('Habilite se esta forma de pagamento é um crediário')
                            ->default(false),

                        Toggle::make('mostrar_no_catalogo')
                            ->label('Mostrar no catálogo')
                            ->helperText('Exibir esta forma de pagamento no catálogo')
                            ->default(false),

                        Toggle::make('lancar_no_contas_receber')
                            ->label('Lançar no contas a receber')
                            ->helperText('Gerar lançamento automático no contas a receber')
                            ->default(false),
                ])->columns(3)
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            // pega apenas as formas de pagamento do tenant do usuario logado
            ->modifyQueryUsing(fn (Builder $query) => $query->where('tenant_id', auth()->user()->tenant_id))
            ->columns([
                Tables\Columns\TextColumn::make('nome')
                    ->searchable(),
                Tables\Columns\IconColumn::make('crediario')
                    ->boolean(),
                Tables\Columns\IconColumn::make('mostrar_no_catalogo')
                    ->boolean(),
                Tables\Columns\IconColumn::make('lancar_no_contas_receber')
                    ->boolean(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'ativo' => 'success',
                        'inativo' => 'danger',
                        default => 'gray',
                    }),
            ])
            ->filters([
                //
            ])
            ->actions([
                EditAction::make()
                    ->tooltip('Editar')
                    ->iconButton()
                    ->color('warning')
                    ->icon('heroicon-o-pencil-square'),
                DeleteAction::make()
                    ->tooltip('Excluir')
                    ->iconButton()
                    ->color('danger')
                    ->icon('heroicon-o-trash')
                    ->modalHeading('Excluir Forma de Pagamento')
                    ->modalDescription('Tem certeza que deseja excluir esta forma de pagamento? Esta ação não poderá ser desfeita.')
                    ->modalSubmitActionLabel('Sim, excluir')
                    ->modalCancelActionLabel('Não, cancelar')
                    ->successNotification(null)
                    ->after(function() {
                        Notification::make()
                            ->success()
                            ->color('success')
                            ->icon('heroicon-o-check-circle')
                            ->title('Forma de Pagamento excluída!')
                            ->body('A forma de pagamento foi removida com sucesso')
                            ->send();
                    })
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->label('Excluir Selecionados')
                        ->modalHeading('Excluir Formas de Pagamento Selecionadas')
                        ->modalDescription('Tem certeza que deseja excluir as formas de pagamento selecionadas? Esta ação não poderá ser desfeita.')
                        ->modalSubmitActionLabel('Sim, excluir')
                        ->modalCancelActionLabel('Não, cancelar')
                        ->successNotification(null)
                        ->after(function() {
                            Notification::make()
                                ->success()
                                ->color('success')
                                ->icon('heroicon-o-check-circle')
                                ->title('Formas de Pagamento excluídas!')
                                ->body('As formas de pagamento foram removidas com sucesso')
                                ->send();
                        })
                ])
                ->label('Ações em massa'),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListFormasPagamento::route('/'),
            'create' => Pages\CreateFormasPagamento::route('/create'),
            'edit' => Pages\EditFormasPagamento::route('/{record}/edit'),
        ];
    }
}
