<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Location: ../index.php');
    exit;
}
require_once '../config/database.php';

// Buscar todos os clientes do usuário atual para o select
$stmt = $pdo->prepare("SELECT id, name FROM clients WHERE user_id = ? ORDER BY name");
$stmt->execute([$_SESSION['user_id']]);
$clients = $stmt->fetchAll();

// Atualizar status das parcelas vencidas
$pdo->query("
    UPDATE installments 
    SET status = 'overdue' 
    WHERE status = 'pending' 
    AND due_date < CURDATE()
");

// Definir variáveis de paginação e busca
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$search = isset($_GET['search']) ? $_GET['search'] : '';
$per_page = 10;
$offset = ($page - 1) * $per_page;

// Buscar total de registros para paginação
$count_query = "SELECT COUNT(DISTINCT m.id) as total 
    FROM movements m 
    JOIN clients c ON m.client_id = c.id 
    WHERE m.user_id = ? ";
$params = [$_SESSION['user_id']];

if (!empty($search)) {
    $count_query .= "AND c.name LIKE ? ";
    $params[] = "%$search%";
}

$stmt = $pdo->prepare($count_query);
$stmt->execute($params);
$total_rows = $stmt->fetch()['total'];
$total_pages = ceil($total_rows / $per_page);

// Buscar movimentos com nome do cliente e status das parcelas
$stmt = $pdo->prepare("
    SELECT 
        m.*,
        c.name as client_name,
        COUNT(DISTINCT i.id) as total_installments,
        SUM(CASE WHEN i.status = 'paid' THEN 1 ELSE 0 END) as paid_installments
    FROM movements m 
    JOIN clients c ON m.client_id = c.id 
    LEFT JOIN installments i ON m.id = i.movement_id
    WHERE m.user_id = ?
    " . (!empty($search) ? "AND c.name LIKE ? " : "") . "
    GROUP BY m.id
    ORDER BY m.movement_date DESC
    LIMIT ? OFFSET ?
");

$query_params = [$_SESSION['user_id']];
if (!empty($search)) {
    $query_params[] = "%$search%";
}
$query_params[] = $per_page;
$query_params[] = $offset;

$stmt->execute($query_params);
$movements = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Movimentos - Sistema Financeiro</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
</head>
<body>
    <?php include '../includes/sidebar.php'; ?>
    <?php include '../includes/navbar.php'; ?>

    <div id="content" class="py-4 px-4">
        <div class="container-fluid">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2>Movimentos</h2>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addMovementModal">
                    <i class="fas fa-plus me-2"></i>Novo Movimento
                </button>
            </div>

            <div class="card">
                <div class="card-body">
                    <!-- Formulário de busca -->
                    <form method="GET" class="mb-4">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="input-group">
                                    <input type="text" name="search" class="form-control" placeholder="Buscar por cliente..." value="<?php echo htmlspecialchars($search); ?>">
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search"></i> Buscar
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>

                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Cliente</th>
                                    <th>Descrição</th>
                                    <th>Valor Total</th>
                                    <th>Parcelas</th>
                                    <th>Valor Parcela</th>
                                    <th>Status</th>
                                    <th>Data do Movimento</th>
                                    <th>Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($movements as $movement): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($movement['client_name']); ?></td>
                                    <td><?php echo htmlspecialchars($movement['description']); ?></td>
                                    <td>R$ <?php echo number_format($movement['total_amount'], 2, ',', '.'); ?></td>
                                    <td><?php echo $movement['paid_installments']; ?>/<?php echo $movement['total_installments']; ?></td>
                                    <td>R$ <?php echo number_format($movement['total_amount'] / $movement['installments'], 2, ',', '.'); ?></td>
                                    <td>
                                        <?php if ($movement['paid_installments'] == $movement['total_installments']): ?>
                                            <span class="badge bg-success">Pago</span>
                                        <?php elseif ($movement['paid_installments'] == 0): ?>
                                            <span class="badge bg-warning">Pendente</span>
                                        <?php else: ?>
                                            <span class="badge bg-info">Parcial</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($movement['movement_date'])); ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-info view-installments" data-id="<?php echo $movement['id']; ?>">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button class="btn btn-sm btn-danger delete-movement" data-id="<?php echo $movement['id']; ?>">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <!-- Paginação -->
                    <?php if ($total_pages > 1): ?>
                    <nav aria-label="Navegação de página" class="mt-4">
                        <ul class="pagination justify-content-center">
                            <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $page-1; ?><?php echo !empty($search) ? '&search='.urlencode($search) : ''; ?>">Anterior</a>
                            </li>
                            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                                <li class="page-item <?php echo $page == $i ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?><?php echo !empty($search) ? '&search='.urlencode($search) : ''; ?>"><?php echo $i; ?></a>
                                </li>
                            <?php endfor; ?>
                            <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                                <a class="page-link" href="?page=<?php echo $page+1; ?><?php echo !empty($search) ? '&search='.urlencode($search) : ''; ?>">Próximo</a>
                            </li>
                        </ul>
                    </nav>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Adicionar Movimento -->
    <div class="modal fade" id="addMovementModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Novo Movimento</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form action="save.php" method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label for="client_id" class="form-label">Cliente</label>
                            <select class="form-select" id="client_id" name="client_id" required>
                                <option value="">Selecione um cliente</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?php echo $client['id']; ?>">
                                        <?php echo htmlspecialchars($client['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label for="description" class="form-label">Descrição</label>
                            <input type="text" class="form-control" id="description" name="description" required>
                        </div>
                        <div class="mb-3">
                            <label for="total_amount" class="form-label">Valor Total</label>
                            <input type="text" class="form-control money-mask" id="total_amount" name="total_amount" required>
                        </div>
                        <div class="mb-3">
                            <label for="movement_date" class="form-label">Data do Movimento</label>
                            <input type="date" class="form-control" id="movement_date" name="movement_date" required value="<?php echo date('Y-m-d'); ?>">
                        </div>
                        <div class="mb-3">
                            <label for="installments" class="form-label">Número de Parcelas</label>
                            <select class="form-select" id="installments" name="installments" required>
                                <?php for ($i = 1; $i <= 12; $i++): ?>
                                    <option value="<?php echo $i; ?>"><?php echo $i; ?>x</option>
                                <?php endfor; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Valor da Parcela</label>
                            <div class="form-control" id="installment_amount">R$ 0,00</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                        <button type="submit" class="btn btn-primary">Salvar</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Modal Ver Parcelas -->
    <div class="modal fade" id="viewInstallmentsModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Parcelas</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="table-responsive">
                        <table class="table" id="installmentsTable">
                            <thead>
                                <tr>
                                    <th>Parcela</th>
                                    <th>Valor</th>
                                    <th>Vencimento</th>
                                    <th>Status</th>
                                    <th>Ações</th>
                                </tr>
                            </thead>
                            <tbody></tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery.mask/1.14.16/jquery.mask.min.js"></script>
    <script>
        $(document).ready(function() {
            // Máscara para valores monetários
            $('.money-mask').mask('#.##0,00', {reverse: true});

            // Calcular valor da parcela
            function calculateInstallment() {
                let total = $('#total_amount').val().replace('.', '').replace(',', '.');
                let installments = $('#installments').val();
                if (total && installments) {
                    let installmentValue = total / installments;
                    $('#installment_amount').text('R$ ' + installmentValue.toLocaleString('pt-BR', { minimumFractionDigits: 2, maximumFractionDigits: 2 }));
                }
            }

            $('#total_amount, #installments').on('input', calculateInstallment);

            // Ver parcelas
            $('.view-installments').click(function() {
                const movementId = $(this).data('id');
                $.get('get_installments.php', { movement_id: movementId }, function(installments) {
                    let tbody = $('#installmentsTable tbody');
                    tbody.empty();
                    
                    installments.forEach(function(inst) {
                        let status = inst.status === 'paid' ? '<span class="badge bg-success">Pago</span>' : 
                                   (inst.status === 'overdue' ? '<span class="badge bg-danger">Atrasado</span>' : 
                                   '<span class="badge bg-warning">Pendente</span>');
                        
                        let actions = inst.status === 'paid' ? '' : 
                            `<button class="btn btn-sm btn-success receive-payment" data-id="${inst.id}">
                                <i class="fas fa-check"></i>
                            </button>`;
                        
                        tbody.append(`
                            <tr>
                                <td>${inst.installment_number}</td>
                                <td>R$ ${parseFloat(inst.amount).toLocaleString('pt-BR', { minimumFractionDigits: 2 })}</td>
                                <td>${new Date(inst.due_date).toLocaleDateString('pt-BR')}</td>
                                <td>${status}</td>
                                <td>${actions}</td>
                            </tr>
                        `);
                    });
                    
                    $('#viewInstallmentsModal').modal('show');
                });
            });

            // Receber pagamento
            $(document).on('click', '.receive-payment', function() {
                const installmentId = $(this).data('id');
                if (confirm('Confirmar recebimento desta parcela?')) {
                    $.post('receive_payment.php', { installment_id: installmentId }, function(response) {
                        if (response.success) {
                            alert('Pagamento registrado com sucesso!');
                            location.reload(); // Recarrega a página para atualizar os totais
                        } else {
                            alert('Erro ao registrar pagamento.');
                        }
                    });
                }
            });

            // Excluir movimento
            $('.delete-movement').click(function() {
                if (confirm('Tem certeza que deseja excluir este movimento e todas as suas parcelas?')) {
                    const id = $(this).data('id');
                    window.location.href = 'delete.php?id=' + id;
                }
            });
        });
    </script>
</body>
</html>
