/**
 * Database Error Handler
 * Gerencia erros de conexão com banco de dados de forma amigável
 */
class DatabaseErrorHandler {
    constructor() {
        this.retryAttempts = 0;
        this.maxRetries = 3;
        this.retryDelay = 5000; // 5 segundos
        this.isShowingError = false;
        
        this.init();
    }

    init() {
        // Intercepta requisições AJAX para detectar erros de banco
        this.interceptAjaxRequests();
        
        // Adiciona listener para erros de rede
        window.addEventListener('online', () => this.handleConnectionRestored());
        window.addEventListener('offline', () => this.handleConnectionLost());
        
        // Verifica periodicamente a conectividade
        this.startHealthCheck();
    }

    interceptAjaxRequests() {
        const originalFetch = window.fetch;
        const self = this;

        window.fetch = function(...args) {
            return originalFetch.apply(this, args)
                .then(response => {
                    if (response.status === 503) {
                        return response.json().then(data => {
                            if (data.error && data.message) {
                                self.handleDatabaseError(data.message);
                                throw new Error(data.message);
                            }
                        });
                    }
                    return response;
                })
                .catch(error => {
                    if (error.message.includes('Failed to fetch') || 
                        error.message.includes('NetworkError') ||
                        error.message.includes('ERR_CONNECTION_REFUSED')) {
                        self.handleDatabaseError('Erro de conexão com o servidor');
                    }
                    throw error;
                });
        };

        // Intercepta jQuery AJAX se disponível
        if (window.jQuery) {
            $(document).ajaxError((event, xhr, settings, error) => {
                if (xhr.status === 503) {
                    try {
                        const response = JSON.parse(xhr.responseText);
                        if (response.error && response.message) {
                            this.handleDatabaseError(response.message);
                        }
                    } catch (e) {
                        this.handleDatabaseError('Erro de conexão com o servidor');
                    }
                }
            });
        }
    }

    handleDatabaseError(message) {
        if (this.isShowingError) return;
        
        this.isShowingError = true;
        this.showErrorModal(message);
    }

    showErrorModal(message) {
        // Remove modal existente se houver
        const existingModal = document.getElementById('database-error-modal');
        if (existingModal) {
            existingModal.remove();
        }

        const modal = document.createElement('div');
        modal.id = 'database-error-modal';
        modal.className = 'modal fade show';
        modal.style.display = 'block';
        modal.style.backgroundColor = 'rgba(0,0,0,0.5)';
        modal.innerHTML = `
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header bg-warning text-dark">
                        <h5 class="modal-title">
                            <i class="ri-database-2-line me-2"></i>
                            Erro de Conexão
                        </h5>
                        <button type="button" class="btn-close" onclick="databaseErrorHandler.closeErrorModal()"></button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-info">
                            <i class="ri-information-line me-2"></i>
                            <strong>Ops! Parece que estamos com problemas técnicos.</strong>
                        </div>
                        <p class="text-muted">
                            ${message || 'Não foi possível conectar ao banco de dados no momento.'}
                        </p>
                        <p class="text-muted">
                            Isso pode acontecer por:
                        </p>
                        <ul class="text-muted">
                            <li>Servidor temporariamente indisponível</li>
                            <li>Manutenção programada</li>
                            <li>Problemas de conectividade</li>
                        </ul>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" onclick="databaseErrorHandler.closeErrorModal()">
                            Fechar
                        </button>
                        <button type="button" class="btn btn-primary" onclick="databaseErrorHandler.retryConnection()">
                            <i class="ri-refresh-line me-2"></i>
                            Tentar Novamente
                        </button>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        
        // Auto-close após 30 segundos
        setTimeout(() => {
            this.closeErrorModal();
        }, 30000);
    }

    closeErrorModal() {
        const modal = document.getElementById('database-error-modal');
        if (modal) {
            modal.remove();
            this.isShowingError = false;
        }
    }

    async retryConnection() {
        this.closeErrorModal();
        
        if (this.retryAttempts >= this.maxRetries) {
            this.showMaxRetriesMessage();
            return;
        }

        this.retryAttempts++;
        
        // Mostra indicador de carregamento
        this.showLoadingIndicator();
        
        try {
            const response = await fetch('/health');
            const data = await response.json();
            
            if (data.status === 'healthy') {
                this.handleConnectionRestored();
                window.location.reload();
            } else {
                throw new Error(data.message);
            }
        } catch (error) {
            this.hideLoadingIndicator();
            
            if (this.retryAttempts < this.maxRetries) {
                setTimeout(() => {
                    this.retryConnection();
                }, this.retryDelay);
            } else {
                this.showMaxRetriesMessage();
            }
        }
    }

    showLoadingIndicator() {
        // Cria indicador de carregamento
        const loading = document.createElement('div');
        loading.id = 'database-loading';
        loading.className = 'position-fixed top-50 start-50 translate-middle';
        loading.style.zIndex = '9999';
        loading.innerHTML = `
            <div class="d-flex align-items-center">
                <div class="spinner-border text-primary me-3" role="status">
                    <span class="visually-hidden">Carregando...</span>
                </div>
                <span class="text-primary">Verificando conexão...</span>
            </div>
        `;
        document.body.appendChild(loading);
    }

    hideLoadingIndicator() {
        const loading = document.getElementById('database-loading');
        if (loading) {
            loading.remove();
        }
    }

    showMaxRetriesMessage() {
        this.showErrorModal('Número máximo de tentativas atingido. Por favor, tente novamente em alguns minutos.');
        this.retryAttempts = 0;
    }

    handleConnectionRestored() {
        this.retryAttempts = 0;
        this.hideLoadingIndicator();
        
        // Mostra notificação de sucesso
        this.showSuccessNotification('Conexão restaurada!');
    }

    handleConnectionLost() {
        this.handleDatabaseError('Conexão com a internet perdida');
    }

    showSuccessNotification(message) {
        // Cria notificação de sucesso
        const notification = document.createElement('div');
        notification.className = 'alert alert-success position-fixed';
        notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
        notification.innerHTML = `
            <i class="ri-check-line me-2"></i>
            ${message}
        `;
        
        document.body.appendChild(notification);
        
        // Remove após 3 segundos
        setTimeout(() => {
            notification.remove();
        }, 3000);
    }

    startHealthCheck() {
        // Verifica a saúde do sistema a cada 5 minutos
        setInterval(async () => {
            try {
                const response = await fetch('/health');
                const data = await response.json();
                
                if (data.status === 'unhealthy') {
                    this.handleDatabaseError(data.message);
                }
            } catch (error) {
                // Silenciosamente ignora erros de health check
            }
        }, 300000); // 5 minutos
    }
}

// Inicializa o handler quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => {
    window.databaseErrorHandler = new DatabaseErrorHandler();
});

// Inicializa imediatamente se o DOM já estiver pronto
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        window.databaseErrorHandler = new DatabaseErrorHandler();
    });
} else {
    window.databaseErrorHandler = new DatabaseErrorHandler();
} 