<?php

namespace App\Services;

error_reporting(E_ALL);
ini_set('display_errors', 'On');

use NFePHP\Common\Certificate;
use NFePHP\NFe\Tools;
use NFePHP\NFe\Make;
use NFePHP\NFe\Common\Standardize;
use NFePHP\NFe\Complements;
use App\Models\Nfce;
use App\Models\Empresa;
use App\Models\ConfiguracaoSuper;
use App\Models\Ibpt;
use App\Models\Contigencia;
use NFePHP\Common\Soap\SoapCurl;
use NFePHP\NFe\Factories\Contingency;
use Illuminate\Support\Facades\Log;

class NFCeService
{

	private $config;
	private $tools;
	protected $empresa_id = null;

	protected $timeout = 8;

	public function __construct($config, $empresa)
	{

		$this->empresa_id = $empresa->id;

		$this->config = $config;
		$this->tools = new Tools(json_encode($config), Certificate::readPfx($empresa->arquivo, $empresa->senha));
		$this->tools->model(65);

		$config = ConfiguracaoSuper::first();
		if($config){
			if($config->timeout_nfce){
				$this->timeout = $config->timeout_nfce;
			}
		}

		$soapCurl = new SoapCurl();
		$soapCurl->httpVersion('1.1');
		$this->tools->loadSoapClass($soapCurl);

		$contigencia = $this->getContigencia();

		if($contigencia != null){
			$contingency = new Contingency($contigencia->status_retorno);
			$this->tools->contingency = $contingency;
		}
		
		// Verificar e criar diretórios necessários
		$xmlDir = public_path('xml_nfce');
		if (!is_dir($xmlDir)) {
			mkdir($xmlDir, 0777, true);
			error_log("NFCeService: Diretório xml_nfce criado: $xmlDir");
		}
		
		$danfeDir = public_path('danfe_temp');
		if (!is_dir($danfeDir)) {
			mkdir($danfeDir, 0777, true);
			error_log("NFCeService: Diretório danfe_temp criado: $danfeDir");
		}
	}

	private function getContigencia(){
		$active = Contigencia::
		where('empresa_id', $this->empresa_id)
		->where('status', 1)
		->where('documento', 'NFCe')
		->first();
		return $active;
	}

	public function gerarXml($item)
	{
		// Verificar se o número já existe e incrementar se necessário
		$numeroOriginal = $item->numero;
		$tentativas = 0;
		while ($tentativas < 10) {
			$nfceExistente = \App\Models\Nfce::where('empresa_id', $item->empresa_id)
				->where('numero', $item->numero)
				->where('estado', '!=', 'cancelado')
				->where('id', '!=', $item->id) // Excluir a própria nota
				->first();
			
			if (!$nfceExistente) {
				break; // Número disponível
			}
			
			$item->numero++; // Incrementa o número
			$tentativas++;
		}
		
		// Se houve mudança no número, salvar
		if ($item->numero != $numeroOriginal) {
			$item->save();
		}

		$nfe = new Make();
		$stdInNFe = new \stdClass();
		$stdInNFe->versao = '4.00';
		$stdInNFe->Id = null;
		$stdInNFe->pk_nItem = '';

		$infNFe = $nfe->taginfNFe($stdInNFe);
		$emitente = $item->empresa;
		$emitente = __objetoParaEmissao($emitente, $item->local_id);

		$cliente = $item->cliente;

		$stdIde = new \stdClass();
		$stdIde->cUF = Empresa::getCodUF($emitente->cidade->uf); // codigo uf emitente
		$stdIde->cNF = $this->gerarCodigoNumerico();
		// $stdIde->natOp = $venda->natureza->natureza;
		$stdIde->natOp = $item->natureza->descricao;

		$stdIde->mod = 65;
		$stdIde->serie = $item->numero_serie;

		// $stdIde->nNF = $item->lastNumero(); // numero sequencial da nfce
		$stdIde->nNF = $item->numero; // numero sequencial da nfe
		$stdIde->dhEmi = date("Y-m-d\TH:i:sP");
		$stdIde->dhSaiEnt = date("Y-m-d\TH:i:sP");
		$stdIde->tpNF = 1;
		$stdIde->idDest = 1;
		$stdIde->cMunFG = $emitente->cidade->codigo;
		$stdIde->tpImp = 4;
		$stdIde->tpEmis = 1;
		$stdIde->cDV = 0;
		$stdIde->tpAmb = (int)$emitente->ambiente;
		$stdIde->finNFe = 1;
		$stdIde->indFinal = 1;
		$stdIde->indPres = 1;
		$stdIde->procEmi = '0';
		$stdIde->verProc = '2.0';
		$tagide = $nfe->tagide($stdIde); //fim da tagide

		// inicia tag do emitente
		$stdEmit = new \stdClass();
		$stdEmit->xNome = $emitente->nome;
		$stdEmit->xFant = $emitente->nome_fantasia;
		$stdEmit->CRT = $emitente->tributacao == 'Regime Normal' ? 3 : 1;
		if($emitente->tributacao == 'Simples Nacional, excesso sublimite de receita bruta'){
			$stdEmit->CRT = 2;	
		}
		$stdEmit->IE = preg_replace('/[^0-9]/', '', $emitente->ie);

		$cpf_cnpj = preg_replace('/[^0-9]/', '', $emitente->cpf_cnpj);
		if (strlen($cpf_cnpj) == 14) {
			$stdEmit->CNPJ = $cpf_cnpj;
		} else {
			$stdEmit->CPF = $cpf_cnpj;
		}
		$emit = $nfe->tagemit($stdEmit);


		$stdEnderEmit = new \stdClass();
		$stdEnderEmit->xLgr = $emitente->rua;
		$stdEnderEmit->nro = $emitente->numero;
		$stdEnderEmit->xCpl = $emitente->complemento;
		$stdEnderEmit->xBairro = $emitente->bairro;
		$stdEnderEmit->cMun = $emitente->cidade->codigo;
		$stdEnderEmit->xMun = $emitente->cidade->nome;
		$stdEnderEmit->UF = $emitente->cidade->uf;
		$stdEnderEmit->CEP = preg_replace('/[^0-9]/', '', $emitente->cep);
		$stdEnderEmit->cPais = '1058';
		$stdEnderEmit->xPais = 'BRASIL';
		$enderEmit = $nfe->tagenderEmit($stdEnderEmit); // fim tag do emitente

		// inicia tag do destinatario
		if ($cliente != null) {

			$stdDest = new \stdClass();
			$stdDest->xNome = $cliente->razao_social;
			if ($cliente->contribuinte == 1) {
				if ($cliente->ie == '') {
					$stdDest->indIEDest = "2";
				} else {
					$stdDest->indIEDest = "1";
				}
			} else {
				$stdDest->indIEDest = "9";
			}

			$cpf_cnpj = preg_replace('/[^0-9]/', '', $cliente->cpf_cnpj);

			if (strlen($cpf_cnpj) == 14) {
				$stdDest->CNPJ = $cpf_cnpj;
				$ie = preg_replace('/[^0-9]/', '', $cliente->ie);
				$stdDest->IE = $ie;
			} else {
				$stdDest->CPF = $cliente->cpf_cnpj;
			}
			$dest = $nfe->tagdest($stdDest);

			$stdEnderDest = new \stdClass();
			$stdEnderDest->xLgr = $cliente->rua;
			$stdEnderDest->nro = $cliente->numero;
			$stdEnderDest->xCpl = $cliente->complemento;
			$stdEnderDest->xBairro = $cliente->bairro;
			$stdEnderDest->cMun = $cliente->cidade->codigo;
			$stdEnderDest->xMun = $cliente->cidade->nome;
			$stdEnderDest->UF = $cliente->cidade->uf;
			$stdEnderDest->fone = preg_replace('/[^0-9]/', '', $cliente->telefone);
			$stdEnderDest->CEP = preg_replace('/[^0-9]/', '', $cliente->cep);
			$stdEnderDest->cPais = '1058';
			$stdEnderDest->xPais = 'BRASIL';

			$enderDest = $nfe->tagenderDest($stdEnderDest);
		}

		if ($item->cliente_cpf_cnpj != "") {
			$stdDest = new \stdClass();
			if ($item->cliente_nome) {
				$stdDest->xNome = $item->cliente_nome;
			}
			$stdDest->indIEDest = "9";

			$doc = preg_replace('/[^0-9]/', '', $item->cliente_cpf_cnpj);
			if (strlen($doc) == 14) $stdDest->CNPJ = $doc;
			else $stdDest->CPF = $doc;
			$dest = $nfe->tagdest($stdDest);
		}
		//fim tag destinatario endereço

		$somaProdutos = 0;
		$somaICMS = 0;
		$somaFrete = 0;
		$somaIpi = 0;
		$totalItens = sizeof($item->itens);
		$somaDesconto = 0;
		$somaAcrescimo = 0;

		$obsIbpt = "";
		$somaFederal = 0;
		$somaEstadual = 0;
		$somaMunicipal = 0;
		$somaVICMSST = 0;

		foreach ($item->itens as $itemCont => $i) {
			$itemCont++;

			$stdProd = new \stdClass(); // tag produto inicio
			$stdProd->item = $itemCont;

			$validaEan = $this->validate_EAN13Barcode($i->produto->codigo_barras);
			$stdProd->cEAN = $validaEan ? $i->produto->codigo_barras : 'SEM GTIN';
			$stdProd->cEANTrib = $validaEan ? $i->produto->codigo_barras : 'SEM GTIN';

			$stdProd->cProd = $i->produto->id;
			$stdProd->xProd = $i->descricao();
			$stdProd->NCM = preg_replace('/[^0-9]/', '', $i->ncm);
			$ibpt = Ibpt::getItemIbpt($emitente->cidade->uf, preg_replace('/[^0-9]/', '', $i->ncm));

			$stdProd->CFOP = $i->cfop;
			$stdProd->uCom = $i->produto->unidade;
			$stdProd->qCom = $i->quantidade;
			$stdProd->vUnCom = $this->format($i->valor_unitario);
			$stdProd->vProd = $this->format(($i->quantidade * $i->valor_unitario));
			$somaProdutos += $i->quantidade * $i->valor_unitario; // Somando o valor total dos produtos
			$stdProd->uTrib = $i->produto->unidade;
			if($i->produto->unidade_tributavel != ''){
				$stdProd->uTrib = $i->produto->unidade_tributavel;
			}
			// $stdProd->qTrib = $i->quantidade;
			if($i->produto->quantidade_tributavel == 0){
				$stdProd->qTrib = $i->quantidade;
			}else{
				$stdProd->qTrib = $i->produto->quantidade_tributavel * $i->quantidade;
			}

			$stdProd->vUnTrib = $this->format($i->valor_unitario);
			if($i->produto->quantidade_tributavel > 0){
				$stdProd->vUnTrib = $stdProd->vProd/$stdProd->qTrib;
			}
			$stdProd->indTot = 1;

			// if ($item->desconto > 0) {
			// 	if ($itemCont < sizeof($item->itens)) {
			// 		$totalVenda = $item->total + $item->desconto;
			// 		$media = (((($stdProd->vProd - $totalVenda) / $totalVenda)) * 100);
			// 		$media = 100 - ($media * -1);
			// 		if ($item->desconto > 0.1) {
			// 			$tempDesc = ($item->desconto * $media) / 100;
			// 		} else {
			// 			$tempDesc = $item->desconto;
			// 		}
			// 		if ($somaDesconto >= $item->desconto) {
			// 			$tempDesc = 0;
			// 		}
			// 		$somaDesconto += $tempDesc;
			// 		if ($tempDesc > 0.01)
			// 			$stdProd->vDesc = $this->format($tempDesc);
			// 	} else {
			// 		if ($item->desconto - $somaDesconto >= 0.01)
			// 			$stdProd->vDesc = $this->format($item->desconto - $somaDesconto);
			// 	}
			// }

			if($item->desconto > 0.01 && $somaDesconto < $item->desconto){
				if($itemCont < sizeof($item->itens)){
					$totalVenda = $item->total + $item->desconto;

					$media = (((($stdProd->vProd - $totalVenda)/$totalVenda))*100);
					$media = 100 - ($media * -1);

					$tempDesc = ($item->desconto*$media)/100;
					$tempDesc -= 0.01;
					if($tempDesc > 0.01){
						$somaDesconto += $this->format($tempDesc);
						$stdProd->vDesc = $this->format($tempDesc);
					}else{
						if(sizeof($item->itens) > 1){
							$somaDesconto += 0.01;
							$stdProd->vDesc = $this->format(0.01);
						}else{
							$somaDesconto = $item->desconto;
							$stdProd->vDesc = $this->format($somaDesconto);
						}
					}
				}else{
					if(($item->desconto - $somaDesconto) > 0.01){
						$stdProd->vDesc = $this->format($item->desconto - $somaDesconto);
					}
				}
			}

			if($item->acrescimo > 0.01 && $somaAcrescimo < $item->acrescimo){
				if($itemCont < sizeof($item->itens)){
					$totalVenda = $item->total;

					$media = (((($stdProd->vProd - $totalVenda)/$totalVenda))*100);
					$media = 100 - ($media * -1);

					$tempDesc = ($item->acrescimo*$media)/100;
					$tempDesc -= 0.01;
					if($tempDesc > 0.01){
						$somaAcrescimo += $this->format($tempDesc);
						$stdProd->vOutro = $this->format($tempDesc);
					}else{
						if(sizeof($item->itens) > 1){
							$somaAcrescimo += 0.01;
							$stdProd->vOutro = $this->format(0.01);
						}else{
							$somaAcrescimo = $item->acrescimo;
							$stdProd->vOutro = $this->format($somaAcrescimo);
						}
					}

				}else{
					if(($item->acrescimo - $somaAcrescimo) > 0.01){
						$stdProd->vOutro = $this->format($item->acrescimo - $somaAcrescimo, 2);
					}
				}
			}

			$prod = $nfe->tagprod($stdProd); // fim tag de produtos

			$stdImposto = new \stdClass();
			$stdImposto->item = $itemCont;

			if($ibpt != null){

				$vProd = $stdProd->vProd;

				if($i->produto->origem == 1 || $i->produto->origem == 2){
					$federal = $this->format(($vProd*($ibpt->importado_federal/100)), 2);

				}else{
					$federal = $this->format(($vProd*($ibpt->nacional_federal/100)), 2);
				}
				$somaFederal += $federal;

				$estadual = $this->format(($vProd*($ibpt->estadual/100)), 2);
				$somaEstadual += $estadual;

				$municipal = $this->format(($vProd*($ibpt->municipal/100)), 2);
				$somaMunicipal += $municipal;

				$soma = $federal + $estadual + $municipal;
				$stdImposto->vTotTrib = $soma;

				$obsIbpt = " FONTE: " . $ibpt->versao ?? '';
				$obsIbpt .= " | ";
			}
			$imposto = $nfe->tagimposto($stdImposto); // tag imposto

			if ($stdEmit->CRT == 1 || $stdEmit->CRT == 4) {
				$stdICMS = new \stdClass();

				$stdICMS->item = $itemCont;
				$stdICMS->orig = 0;
				$stdICMS->CSOSN = $i->cst_csosn;

				// Tratamento especial para CSOSN 900
				if($i->cst_csosn == 900) {
					$stdICMS->modBC = 0;
					$stdICMS->pICMS = $this->format($i->perc_icms);
					$stdICMS->vICMS = ($stdProd->vProd - $stdProd->vDesc) * ($stdICMS->pICMS/100);
					
					// Se tiver redução da base de cálculo
					if($i->perc_red_bc > 0){
						$stdICMS->pRedBC = $this->format($i->perc_red_bc);
					}
				}
				// Apenas adicionar pCredSN se não for CSOSN 900
				else if($i->cst_csosn != 900) {
					$stdICMS->pCredSN = $this->format($i->perc_icms);
					$stdICMS->vCredICMSSN = $this->format($i->perc_icms);
				}

				if($i->cst_csosn == 61){
					$stdICMS->CST = $i->cst_csosn;
					$stdICMS->qBCMonoRet = $this->format($stdProd->qTrib);
					$stdICMS->adRemICMSRet = $this->format($i->produto->adRemICMSRet, 4);
					$stdICMS->vICMSMonoRet = $this->format($i->produto->adRemICMSRet*$stdProd->qTrib, 4);
					$ICMS = $nfe->tagICMS($stdICMS);
				}else{
					$ICMS = $nfe->tagICMSSN($stdICMS);
				}
			} else if ($stdEmit->CRT == 3 || $stdEmit->CRT == 2) {


				$stdICMS = new \stdClass();
				$stdICMS->item = $itemCont;
				$stdICMS->orig = 0;
				$stdICMS->CST = $i->cst_csosn;
				$stdICMS->modBC = 0;
				$stdICMS->vBC = $stdProd->vProd - $stdProd->vDesc;
				$stdICMS->pICMS = $this->format($i->perc_icms);
				$stdICMS->vICMS = $stdICMS->vBC * ($stdICMS->pICMS / 100);

				if($i->cst_csosn == '60'){
					$stdICMS->vBCSTRet = 0.00;
					$stdICMS->vICMSSTRet = 0.00;
					$stdICMS->vBCSTDest = 0.00;
					$stdICMS->vICMSSTDest = 0.00;
					$stdICMS->pST = 0.00;
				}

				if($i->cst_csosn != 60){
					if($stdICMS->vICMS > 0){
						$somaICMS += $stdICMS->vICMS;
					}
				}
				if($i->cst_csosn == 10){

					$stdICMS->modBCST = $i->produto->modBCST;
					$stdICMS->vBCST = $stdProd->vProd;
					$stdICMS->pICMSST = $this->format($i->produto->pICMSST);
					$somaVICMSST += $stdICMS->vICMSST = $stdICMS->vBCST * ($stdICMS->pICMSST/100);
				}

				if($i->cst_csosn == 61){

					$stdICMS->qBCMonoRet = $this->format($stdProd->qTrib);
					$stdICMS->adRemICMSRet = $this->format($i->produto->adRemICMSRet, 4);
					$stdICMS->vICMSMonoRet = $this->format($i->produto->adRemICMSRet*$stdProd->qTrib, 4);
				}

				// $ICMS = $nfe->tagICMS($stdICMS);
				if($i->cst_csosn == '60'){
					$ICMS = $nfe->tagICMSST($stdICMS);
				}else{
					$ICMS = $nfe->tagICMS($stdICMS);
				}
			} // fim tag icms
			//PIS
			$stdPIS = new \stdClass();
			$stdPIS->item = $itemCont;
			$stdPIS->CST = $i->cst_pis;
			$stdPIS->vBC = $this->format($i->perc_pis) > 0 ? $stdProd->vProd : 0.00;
			$stdPIS->pPIS = $this->format($i->perc_pis);
			$stdPIS->vPIS = $this->format($stdProd->vProd * ($i->perc_pis / 100));
			$PIS = $nfe->tagPIS($stdPIS);

			//COFINS
			$stdCOFINS = new \stdClass();
			$stdCOFINS->item = $itemCont;
			$stdCOFINS->CST = $i->cst_cofins;
			$stdCOFINS->vBC = $this->format($i->cst_cofins) > 0 ? $stdProd->vProd : 0.00;
			$stdCOFINS->pCOFINS = $this->format($i->perc_cofins);
			$stdCOFINS->vCOFINS = $this->format($stdProd->vProd * ($i->perc_cofins / 100));
			$COFINS = $nfe->tagCOFINS($stdCOFINS);

			if(strlen($i->produto->codigo_anp) > 2){
				$stdComb = new \stdClass();
				$stdComb->item = $itemCont; 
				$stdComb->cProdANP = $i->produto->codigo_anp;
				$stdComb->descANP = $i->produto->getDescricaoAnp();
				if($i->produto->perc_glp > 0){
					$stdComb->pGLP = $this->format($i->produto->perc_glp);
				}

				if($i->produto->perc_gnn > 0){
					$stdComb->pGNn = $this->format($i->produto->perc_gnn);
				}

				if($i->produto->perc_gni > 0){
					$stdComb->pGNi = $this->format($i->produto->perc_gni);
				}

				$stdComb->vPart = $this->format($i->produto->valor_partida);
				$stdComb->UFCons = $item->cliente ? $item->cliente->cidade->uf : $emitente->cidade->uf;

				if($i->produto->pBio > 0){
					$stdComb->pBio = $i->produto->pBio;
				}
				$nfe->tagcomb($stdComb);
			}

			if($stdIde->indFinal == 0 && strlen($i->produto->codigo_anp) > 2){
				$stdOrigComb = new \stdClass();

				$stdOrigComb->item = $itemCont; 
				$stdOrigComb->indImport = $i->produto->indImport;
				$stdOrigComb->cUFOrig = $i->produto->cUFOrig;
				$stdOrigComb->pOrig = $i->produto->pOrig;
				$nfe->tagorigComb($stdOrigComb);
			}

			$cest = $i->produto->cest;
			$cest = str_replace(".", "", $cest);
			$stdProd->CEST = $cest;
			if(strlen($cest) > 0){
				$std = new \stdClass();
				$std->item = $itemCont; 
				$std->CEST = $cest;
				$nfe->tagCEST($std);
			}
		}

		$stdICMSTot = new \stdClass();
		$stdICMSTot->vBC = $stdEmit->CRT == 3 ? $this->format($somaProdutos) : 0.00;
		$stdICMSTot->vICMS = $this->format($somaICMS);
		$stdICMSTot->vICMSDeson = 0.00;
		$stdICMSTot->vBCST = 0.00;
		$stdICMSTot->vST = 0.00;
		$stdICMSTot->vProd = $this->format($somaProdutos);
		$stdICMSTot->vFrete = 0.00;
		$stdICMSTot->vSeg = 0.00;
		
		// Garante que o desconto seja considerado corretamente
		$desconto = ($item->desconto && $item->desconto > 0) ? $item->desconto : 0.00;
		$stdICMSTot->vDesc = $this->format($desconto);
		
		$stdICMSTot->vII = 0.00;
		$stdICMSTot->vIPI = 0.00;
		$stdICMSTot->vPIS = 0.00;
		$stdICMSTot->vCOFINS = 0.00;
		$stdICMSTot->vOutro = 0.00;
		
		// Calcula o valor total da NFCe considerando o desconto
		$valorTotal = $somaProdutos - $desconto + $item->acrescimo;
		$stdICMSTot->vNF = $this->format($valorTotal);

		// Log para diagnóstico
		Log::info('Valores calculados para XML:', [
			'somaProdutos' => $somaProdutos,
			'desconto' => $desconto,
			'acrescimo' => $item->acrescimo,
			'valorTotal' => $valorTotal
		]);

		$stdICMSTot->vTotTrib = 0.00;
		$ICMSTot = $nfe->tagICMSTot($stdICMSTot);

		$stdTransp = new \stdClass();
		$stdTransp->modFrete = 9;

		$transp = $nfe->tagtransp($stdTransp);

		$respTec = ConfiguracaoSuper::first();
		if ($respTec != null && $respTec->usar_resp_tecnico == 1) {
			$stdResp = new \stdClass();
			$doc = preg_replace('/[^0-9]/', '', $respTec->cpf_cnpj);
			if (strlen($doc) == 14) $stdResp->CNPJ = $doc;
			else $stdResp->CPF = $doc;

			$stdResp->xContato = $respTec->name;
			$stdResp->email = $respTec->email;
			$stdResp->fone = preg_replace('/[^0-9]/', '', $respTec->telefone);
			$nfe->taginfRespTec($stdResp);
		}

		// dd($item->desconto);
		//Fatura
		$stdFat = new \stdClass();
		$stdFat->nFat = $stdIde->nNF;
		$stdFat->vOrig = $this->format($item->total);
		$stdFat->vDesc = $this->format($item->desconto);
		$stdFat->vLiq = $this->format($item->total);

		// $fatura = $nfe->tagfat($stdFat);

		$contFatura = 1;

		$stdPag = new \stdClass();
		if ($item->dinheiro_recebido > 0) {
			$vPag = $item->dinheiro_recebido;
			$stdPag->vTroco = $vPag - $item->total;
		}
		// dd($this->format($item->total));
		$pag = $nfe->tagpag($stdPag);


		if ($item->dinheiro_recebido > 0) {
			$stdDetPag = new \stdClass();
			$stdDetPag->tPag = $item->tipo_pagamento;
			if($item->tipo_pagamento == '06'){
				$stdDetPag->tPag = '05'; 
			}
			$stdDetPag->vPag = $this->format($item->dinheiro_recebido);
			$stdDetPag->indPag = 1;
			$detPag = $nfe->tagdetPag($stdDetPag);
		} else {

			if (sizeof($item->fatura) > 0) {
				// NOVA FUNCIONALIDADE: Agrupar pagamentos do mesmo tipo
				$pagamentosAgrupados = [];

				// Agrupar os pagamentos por tipo
				foreach ($item->fatura as $ft) {
					$tipoPagamento = $ft->tipo_pagamento;
					
					// Normalizar tipos de pagamento
					if($tipoPagamento == '30'){
						$tipoPagamento = '03';
					}elseif($tipoPagamento == '31'){
						$tipoPagamento = '04';
					}elseif($tipoPagamento == '32'){
						$tipoPagamento = '17';
					}
					
					// Regra especial para pagamento tipo '06' - convertido para '05'
					if($tipoPagamento == '06'){
						$tipoPagamento = '05'; 
					}
					
					// Se não existir este tipo no array de pagamentos agrupados, inicializa
					if (!isset($pagamentosAgrupados[$tipoPagamento])) {
						$pagamentosAgrupados[$tipoPagamento] = [
							'tipo_pagamento' => $tipoPagamento,
							'valor' => 0,
							'bandeira_cartao' => $item->bandeira_cartao,
							'cnpj_cartao' => $item->cnpj_cartao,
							'cAut_cartao' => $item->cAut_cartao
						];
					}
					
					// Soma o valor ao grupo correspondente
					$pagamentosAgrupados[$tipoPagamento]['valor'] += $ft->valor;
				}
				
				// Adicionar cada grupo de pagamento ao XML
				foreach ($pagamentosAgrupados as $pag) {
					$stdDetPag = new \stdClass();
					$stdDetPag->tPag = $pag['tipo_pagamento'];
					$stdDetPag->vPag = $this->format($pag['valor']);
					$stdDetPag->indPag = 1;
					
					// Adicionar informações específicas para pagamentos com cartão
					if($pag['tipo_pagamento'] == '03' || $pag['tipo_pagamento'] == '04' || $pag['tipo_pagamento'] == '17') {
						$stdDetPag->tBand = $pag['bandeira_cartao'];
						if(!$pag['bandeira_cartao']){
							$stdDetPag->tBand = '01';
						}
						
						if($pag['cnpj_cartao']){
							$stdDetPag->CNPJ = preg_replace('/[^0-9]/', '', $pag['cnpj_cartao']);
						}
						
						if($pag['cAut_cartao'] != ""){
							$stdDetPag->cAut = $pag['cAut_cartao'];
						}
						
						$stdDetPag->tpIntegra = 2;
					}
					
					$detPag = $nfe->tagdetPag($stdDetPag);
				}
			}
		}

		$stdInfoAdic = new \stdClass();
		$obs = $item->observacao;

		if($somaEstadual > 0 || $somaFederal > 0 || $somaMunicipal > 0){
			$obs .= " Trib. aprox. ";
			if($somaFederal > 0){
				$obs .= "R$ " . number_format($somaFederal, 2, ',', '.') ." Federal"; 
			}
			if($somaEstadual > 0){
				$obs .= ", R$ ".number_format($somaEstadual, 2, ',', '.')." Estadual"; 
			}
			if($somaMunicipal > 0){
				$obs .= ", R$ ".number_format($somaMunicipal, 2, ',', '.')." Municipal"; 
			}
			// $ibpt = IBPT::where('uf', $config->UF)->first();

			$obs .= $obsIbpt;
		}
		if(trim($emitente->observacao_padrao_nfce) != ""){
			$obs .= " " . $emitente->observacao_padrao_nfce;
		}

		$stdInfoAdic->infCpl = $obs;

		$infoAdic = $nfe->taginfAdic($stdInfoAdic);

		try {
			$nfe->montaNFe();
			$arr = [
				'chave' => $nfe->getChave(),
				'xml' => $nfe->getXML(),
				'numero' => $stdIde->nNF
			];
			return $arr;
		} catch (\Exception $e) {
			return [
				'erros_xml' => $nfe->getErrors()
			];
		}
	}

	private function validate_EAN13Barcode($ean)
	{

		$sumEvenIndexes = 0;
		$sumOddIndexes  = 0;

		$eanAsArray = array_map('intval', str_split($ean));

		if(strlen($ean) == 14){
			return true;
		}

		if (!$this->has13Numbers($eanAsArray) ) {
			return false;
		};

		for ($i = 0; $i < count($eanAsArray)-1; $i++) {
			if ($i % 2 === 0) {
				$sumOddIndexes  += $eanAsArray[$i];
			} else {
				$sumEvenIndexes += $eanAsArray[$i];
			}
		}

		$rest = ($sumOddIndexes + (3 * $sumEvenIndexes)) % 10;

		if ($rest !== 0) {
			$rest = 10 - $rest;
		}

		return $rest === $eanAsArray[12];
	}

	private function has13Numbers(array $ean)
	{
		return count($ean) === 13 || count($ean) === 14;
	}

	public function format($number, $dec = 2)
	{
		return number_format((float) $number, $dec, ".", "");
	}

	public function sign($xml)
	{
		return $this->tools->signNFe($xml);
	}

	public function transmitir($signXml, $chave)
	{
		// Log simples para debug
		file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - NFCeService transmitir iniciado para chave: $chave\n", FILE_APPEND);
		
		try {
			$idLote = str_pad(100, 15, '0', STR_PAD_LEFT);
			$indSinc = 1; // Sempre síncrono para 1 NF-e
			$resp = $this->tools->sefazEnviaLote([$signXml], $idLote, $indSinc);

			$st = new Standardize();
			$std = $st->toStd($resp);
			sleep($this->timeout);
			
			// Log do resultado do envio
			file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Resultado envio: cStat=" . $std->cStat . ", xMotivo=" . $std->xMotivo . "\n", FILE_APPEND);
			
			// Aceitar tanto cStat 103 (Lote recebido) quanto 104 (Lote processado)
			if ($std->cStat != 103 && $std->cStat != 104) {
				file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Erro no envio: " . $std->cStat . " - " . $std->xMotivo . "\n", FILE_APPEND);
				return [
					'erro' => 1,
					'error' => "[$std->cStat] - $std->xMotivo"
				];
			}
			
			// Verificar se tem recibo (cStat 103) ou se já foi processado (cStat 104)
			if ($std->cStat == 103 && isset($std->infRec)) {
				$recibo = $std->infRec->nRec;
				$temRecibo = true;
			} else {
				// Se cStat 104, não tem recibo, mas vamos tentar buscar o protocolo pela chave
				$temRecibo = false;
			}

			// Loop de consulta ao protocolo
			$tentativas = 5;
			for ($i = 0; $i < $tentativas; $i++) {
				file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Tentativa $i de consulta do protocolo\n", FILE_APPEND);
				
				if ($temRecibo) {
					// Se tem recibo, consulta pelo recibo
					$protocolo = $this->tools->sefazConsultaRecibo($recibo);
					file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Consulta por recibo: $recibo\n", FILE_APPEND);
				} else {
					// Se não tem recibo, consulta pela chave
					$protocolo = $this->tools->sefazConsultaChave($chave);
					file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Consulta por chave: $chave\n", FILE_APPEND);
				}
				
				$stProtocolo = new Standardize();
				$stdProtocolo = $stProtocolo->toStd($protocolo);
				
				// Log para debug da estrutura da resposta
				file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Estrutura da resposta: " . print_r($stdProtocolo, true) . "\n", FILE_APPEND);
				error_log("NFCeService: Estrutura da resposta - Tem recibo: " . ($temRecibo ? 'Sim' : 'Não'));
				error_log("NFCeService: Resposta completa: " . print_r($stdProtocolo, true));
				
				// Verificar se a consulta pela chave retornou autorização
				$cStat = null;
				$xMotivo = null;
				
				if ($temRecibo) {
					// Consulta pelo recibo - estrutura normal
					if (isset($stdProtocolo->protNFe->infProt->cStat)) {
						$cStat = $stdProtocolo->protNFe->infProt->cStat;
						$xMotivo = $stdProtocolo->protNFe->infProt->xMotivo;
						error_log("NFCeService: Consulta por recibo - cStat: $cStat, xMotivo: $xMotivo");
					}
				} else {
					// Consulta pela chave - estrutura diferente
					if (isset($stdProtocolo->cStat)) {
						$cStat = $stdProtocolo->cStat;
						$xMotivo = $stdProtocolo->xMotivo;
						error_log("NFCeService: Consulta por chave - cStat: $cStat, xMotivo: $xMotivo");
					}
				}
				
				// Verificar se foi autorizado (cStat 100) ou se já está autorizada
				if ($cStat == 100 || 
					($cStat == 104 && strpos(strtolower($xMotivo), 'autorizado') !== false) ||
					($cStat == 104 && strpos(strtolower($xMotivo), 'processado') !== false) ||
					($cStat == 104 && strpos(strtolower($xMotivo), 'sucesso') !== false)) {
					
					file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - NFCe autorizada! cStat: $cStat, xMotivo: $xMotivo\n", FILE_APPEND);
					
					try {
						$xml = Complements::toAuthorize($signXml, $protocolo);
						// Recupera o CNPJ e nome da empresa da nota
						$empresa = \App\Models\Empresa::findOrFail($this->empresa_id);
						$cnpj = preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj);
						$nomeEmpresa = $empresa->nome;
						// Recupera mês e ano da emissão da chave da NFCe
						// A chave tem 44 dígitos: 2(UF) + 2(AA) + 2(MM) + 14(CNPJ) + 2(MODELO) + 3(SERIE) + 9(NUMERO) + 1(DV) + 9(CODIGO)
						$ano = '20' . substr($chave, 2, 2); // Ano da emissão
						$mes = substr($chave, 4, 2); // Mês da emissão
						$mesAno = $mes . $ano;
						// Monta o diretório
						$dir = public_path('xml_nfce/' . $cnpj . '/' . $mesAno . '/');
						if (!is_dir($dir)) {
							mkdir($dir, 0777, true);
						}
						$xmlPath = $dir . $chave . '.xml';
						// LOG DE DEPURAÇÃO DO SALVAMENTO DE XML
						file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Salvando XML: EMPRESA: $nomeEmpresa, CNPJ=$cnpj, mesAno=$mesAno, caminho=$xmlPath\n", FILE_APPEND);
						file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Diretório existe? " . (is_dir($dir) ? 'Sim' : 'Não') . "\n", FILE_APPEND);
						$salvou = file_put_contents($xmlPath, $xml);
						file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Resultado do file_put_contents: " . ($salvou !== false ? 'Sucesso' : 'Falha') . "\n", FILE_APPEND);
						file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - XML existe após salvar: " . (file_exists($xmlPath) ? 'Sim' : 'Não') . "\n", FILE_APPEND);
						
						// Log para debug
						error_log("NFCeService: Tentativa de salvar XML - Chave: $chave, Resultado: " . ($salvou !== false ? 'Sucesso' : 'Falha'));
						error_log("NFCeService: Caminho do XML: $xmlPath");
						error_log("NFCeService: XML existe após salvar: " . (file_exists($xmlPath) ? 'Sim' : 'Não'));
						
						if ($salvou !== false) {
							file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - XML salvo com sucesso!\n", FILE_APPEND);
							return [
								'erro' => 0,
								// Ajuste: retorna o nProt se existir, senão o recibo, senão mensagem padrão
								'success' => isset($stdProtocolo->protNFe->infProt->nProt) ? $stdProtocolo->protNFe->infProt->nProt : ($temRecibo ? $recibo : 'Lote processado com sucesso')
							];
						} else {
							file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Erro ao salvar XML!\n", FILE_APPEND);
							return [
								'erro' => 1,
								'error' => 'Erro ao salvar XML autorizado',
								'recibo' => $temRecibo ? $recibo : null
							];
						}
					} catch (\Exception $e) {
						file_put_contents(storage_path('logs/nfce_debug.log'), date('Y-m-d H:i:s') . " - Exceção ao salvar XML: " . $e->getMessage() . "\n", FILE_APPEND);
						error_log("NFCeService: Exceção ao salvar XML - " . $e->getMessage());
						return [
							'erro' => 1,
							'error' => $stProtocolo->toArray($protocolo),
							'recibo' => $temRecibo ? $recibo : null
						];
					}
				}
				
				// Se a consulta pela chave retorna que a NFCe não foi encontrada, pode estar sendo processada
				if ($cStat == 104 && strpos(strtolower($xMotivo), 'não encontrada') !== false) {
					error_log("NFCeService: NFCe não encontrada na consulta pela chave - pode estar sendo processada");
					// Continua o loop para tentar novamente
				}
				
				// Se a consulta pela chave retorna outros códigos que indicam processamento
				if ($cStat == 104 && (
					strpos(strtolower($xMotivo), 'processando') !== false ||
					strpos(strtolower($xMotivo), 'aguardando') !== false ||
					strpos(strtolower($xMotivo), 'pendente') !== false
				)) {
					error_log("NFCeService: NFCe ainda sendo processada - cStat: $cStat, xMotivo: $xMotivo");
					// Continua o loop para tentar novamente
				}
				
				// Se não for autorizado, espera 3 segundos e tenta de novo
				sleep(3);
			}
			
			// Se não autorizou após as tentativas
			$motivo = $xMotivo ?: 'Motivo não informado';
			$cStatFinal = $cStat ?: 'Desconhecido';
			return [
				'erro' => 1,
				'error' => "[$cStatFinal] - $motivo (Autorização pendente, tente novamente em instantes)",
				'recibo' => $temRecibo ? $recibo : null
			];
		} catch(\Exception $e){
			error_log("NFCeService: Exceção geral - " . $e->getMessage());
			return [
				'erro' => 1,
				'error' => $e->getMessage()
			];
		}
	} 

	public function consultar($nfe)
	{
		try {

			$this->tools->model('65');

			$chave = $nfe->chave;
			$response = $this->tools->sefazConsultaChave($chave);

			$stdCl = new Standardize($response);
			$arr = $stdCl->toArray();
			if($arr['xMotivo'] == 'Autorizado o uso da NF-e'){
				if($nfe->estado != 'aprovado'){

					$empresa = Empresa::findOrFail($nfe->empresa_id);
					$empresa = __objetoParaEmissao($empresa, $nfe->local_id);

					$chave = $arr['protNFe']['infProt']['chNFe'];
					$nRec = $nfe->recibo;
					$nfe->estado = 'aprovado';
					$nfe->save();

					if($empresa->ambiente == 1){
						$empresa->numero_ultima_nfce_producao = $nfe->numero;
					}else{
						$empresa->numero_ultima_nfce_homologacao = $nfe->numero;
					}
					try{
						$xml = Complements::toAuthorize($nfe->signed_xml, $nRec);
						// Organiza o XML por CNPJ/mês-ano
						$cnpj = preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj);
						$ano = '20' . substr($chave, 2, 2); // Ano da emissão
						$mes = substr($chave, 4, 2); // Mês da emissão
						$mesAno = $mes . $ano;
						$dir = public_path('xml_nfce/' . $cnpj . '/' . $mesAno . '/');
						if (!is_dir($dir)) {
							mkdir($dir, 0777, true);
						}
						file_put_contents($dir . $chave . '.xml', $xml);
					}catch(\Exception $e){
						
					}

				}
			}

			return $arr;
		} catch (\Exception $e) {
			return ['erro' => true, 'data' => $e->getMessage(), 'status' => 402];
		}
	}

	public function cancelar($nfe, $motivo)
	{
		try {
			// Força o timezone para Brasil/São Paulo para evitar problemas de fuso horário
			date_default_timezone_set('America/Fortaleza');

			$chave = $nfe->chave;
			$response = $this->tools->sefazConsultaChave($chave);
			$stdCl = new Standardize($response);
			$arr = $stdCl->toArray();
			sleep(1);

			$nProt = $arr['protNFe']['infProt']['nProt'];

			// Usar hora de emissão da NFCe como base para o cancelamento
			// Adiciona alguns segundos para evitar problemas de sincronização
			$dataEmissao = null;
			if (isset($nfe->data_emissao)) {
				$dataEmissao = new \DateTime($nfe->data_emissao);
				$dataEmissao->modify('+30 seconds'); // Adiciona 30 segundos
			}

			$response = $this->tools->sefazCancela($chave, $motivo, $nProt, $dataEmissao);
			sleep(2);
			$stdCl = new Standardize($response);
			$std = $stdCl->toStd();
			$arr = $stdCl->toArray();
			$json = $stdCl->toJson();

			if ($std->cStat != 128) {
				//TRATAR
			} else {
				$cStat = $std->retEvento->infEvento->cStat;
				if ($cStat == '101' || $cStat == '135' || $cStat == '155') {
					$xml = Complements::toAuthorize($this->tools->lastRequest, $response);
					file_put_contents(public_path('xml_nfce_cancelada/') . $chave . '.xml', $xml);

					return $arr;
				} else {

					return ['erro' => true, 'data' => $arr, 'status' => 402];
				}
			}
		} catch (\Exception $e) {
			// echo $e->getMessage();
			return ['erro' => true, 'data' => $e->getMessage(), 'status' => 402];
			//TRATAR
		}
	}

	public function consultaStatus($tpAmb, $uf)
	{
		try {
			$response = $this->tools->sefazStatus($uf, $tpAmb);
			$stdCl = new Standardize($response);
			$arr = $stdCl->toArray();
			return $arr;
		} catch (\Exception $e) {
			echo $e->getMessage();
		}
	}

	private function gerarCodigoNumerico()
	{
		// Gera um código numérico único baseado no timestamp e número aleatório
		// Isso garante que não haja conflitos de chave de acesso
		$timestamp = time();
		$random = mt_rand(1000, 9999);
		$codigo = ($timestamp % 100000) * 10000 + $random;
		
		// Garante que o código tenha exatamente 8 dígitos
		$codigoFinal = str_pad($codigo % 100000000, 8, '0', STR_PAD_LEFT);
		
		return $codigoFinal;
	}
}
