/**
 * Dashboard Ultramoderno
 * Arquivo JavaScript para implementar funcionalidades interativas e dinâmicas ao dashboard
 * 
 * IMPORTANTE: Este dashboard utiliza APENAS dados reais obtidos diretamente do banco de dados.
 * Todos os valores são calculados no backend (DashboardController) e passados para esta interface.
 * Não há geração de dados aleatórios.
 */

document.addEventListener('DOMContentLoaded', function() {
    // Inicializa componentes do dashboard
    initializeDashboard();
    initializeAnimatedCounters();
    setupCharts();
    setupDateFilters();
    setupScrollAnimations();
    initializeBootstrapComponents();
    setupResponsiveLayout();
    
    // Definir período padrão como "Este Mês"
    const periodoDefault = 'month';
    const periodoAtual = document.getElementById('inp-periodo') ? 
        document.getElementById('inp-periodo').value : periodoDefault;
    
    // Se não houver um período específico na URL, usar 'month' (Este Mês)
    if (periodoAtual === 'today') {
        // Encontrar e ativar o botão "Este Mês"
        const monthButton = document.querySelector('.date-range-btn[data-range="month"]');
        if (monthButton) {
            // Remover classe active de todos os botões
            document.querySelectorAll('.date-range-btn').forEach(btn => btn.classList.remove('active'));
            // Adicionar classe active ao botão do mês
            monthButton.classList.add('active');
            
            // Atualizar o input escondido
            if (document.getElementById('inp-periodo')) {
                document.getElementById('inp-periodo').value = periodoDefault;
            }
            
            // Esconder o input de data personalizada se estiver visível
            const customDateRange = document.getElementById('customDateRange');
            if (customDateRange) {
                customDateRange.classList.remove('visible');
                customDateRange.classList.add('hidden');
            }
            
            // Carregar dados do mês atual
            updateCardFootersByPeriod(periodoDefault);
            updateDashboardData(periodoDefault);
        }
    }
});

/**
 * Inicializa os componentes básicos do dashboard
 */
function initializeDashboard() {
    // Ativa os tooltips e popovers do Bootstrap
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl, {
            boundary: document.body
        });
    });

    // Adiciona comportamento aos checkboxes das tarefas
    document.querySelectorAll('.form-check-input').forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            const listItem = this.closest('.list-group-item');
            if (this.checked) {
                listItem.classList.add('completed-task');
                listItem.style.opacity = '0.7';
            } else {
                listItem.classList.remove('completed-task');
                listItem.style.opacity = '1';
            }
        });
    });

    // Inicializa ações dos headers dos cards
    document.querySelectorAll('.card-header-action').forEach(action => {
        action.addEventListener('click', function() {
            if (this.querySelector('.ri-refresh-line')) {
                refreshCardData(this.closest('.card'));
            } else if (this.querySelector('.ri-fullscreen-line')) {
                toggleFullscreen(this.closest('.card'));
            } else if (this.querySelector('.ri-more-2-fill')) {
                showCardOptions(this);
            }
        });
    });
}

/**
 * Inicializa os contadores animados
 */
function initializeAnimatedCounters() {
    const counters = document.querySelectorAll('.stat-value, .card-value');
    
    counters.forEach(counter => {
        // Pula se não for um formato numérico
        if (counter.textContent.includes('h') || counter.textContent.includes('%')) return;
        
        const value = counter.textContent;
        let startValue = 0;
        let endValue;
        
        // Extrai o valor numérico 
        if (value.includes('R$')) {
            // Valor monetário
            endValue = parseFloat(value.replace('R$', '').replace(/\./g, '').replace(',', '.').trim());
            counter.textContent = 'R$ 0';
        } else {
            // Valor numérico simples
            endValue = parseInt(value.replace(/\D/g, ''));
            counter.textContent = '0';
        }
        
        const duration = 1500;
        const increment = endValue / (duration / 16); // 60fps
        
        function updateCounter() {
            startValue += increment;
            
            if (startValue < endValue) {
                if (value.includes('R$')) {
                    counter.textContent = 'R$ ' + formatMoney(startValue);
                } else {
                    counter.textContent = Math.floor(startValue).toString();
                }
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = value; // Restaura o valor original
            }
        }
        
        updateCounter();
    });
    
    // Inicia sparklines para gráficos pequenos
    document.querySelectorAll('[data-sparkline]').forEach(element => {
        const values = element.getAttribute('data-sparkline').split(',').map(v => parseInt(v));
        createSparkline(element, values);
    });
}

/**
 * Cria um sparkline usando ApexCharts
 */
function createSparkline(element, data) {
    const colors = {
        'primary': '#4361ee',
        'success': '#2d9b4e',
        'warning': '#ff9800',
        'danger': '#e53935'
    };
    
    // Detecta a cor baseada nas classes do card
    let color = '#4361ee'; // cor padrão
    const card = element.closest('.stat-card-ultramodern');
    
    if (card) {
        Object.keys(colors).forEach(key => {
            if (card.classList.contains(key)) {
                color = colors[key];
            }
        });
    }
    
    const options = {
        series: [{
            data: data
        }],
        chart: {
            type: 'line',
            width: 100,
            height: 35,
            sparkline: {
                enabled: true
            }
        },
        stroke: {
            curve: 'smooth',
            width: 2
        },
        colors: [color],
        tooltip: {
            fixed: {
                enabled: false
            },
            x: {
                show: false
            },
            y: {
                title: {
                    formatter: function() {
                        return '';
                    }
                }
            },
            marker: {
                show: false
            }
        }
    };
    
    new ApexCharts(element, options).render();
}

/**
 * Configura os gráficos do dashboard
 * 
 * IMPORTANTE: Todos os gráficos são inicializados com dados reais do banco.
 * Os dados são passados do controller para a view como atributos data-* nos elementos HTML.
 * Sem valores hardcoded ou aleatórios.
 */
function setupCharts() {
    if (document.getElementById('salesChart')) {
        const chartElement = document.getElementById('salesChart');
        let salesData = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]; // Valores padrão
        
        // Verificar se há dados reais disponíveis
        if (chartElement.dataset.sales) {
            try {
                // Garantir que os dados sejam um array válido
                const parsedData = JSON.parse(chartElement.dataset.sales);
                if (Array.isArray(parsedData)) {
                    salesData = parsedData;
                } else if (typeof parsedData === 'object') {
                    // Se for um objeto, extrair os valores em um array
                    salesData = Object.values(parsedData);
                }
            } catch (e) {
                console.error('Erro ao parsear dados de vendas:', e);
            }
        }
        
        const salesChartOptions = {
            series: [{
                name: 'Vendas',
                data: salesData
            }],
            chart: {
                height: 350,
                type: 'area',
                fontFamily: 'inherit',
                dropShadow: {
                    enabled: true,
                    opacity: 0.1,
                    blur: 5,
                    left: -7,
                    top: 5
                },
                toolbar: {
                    show: false
                },
                animations: {
                    enabled: true,
                    easing: 'easeinout',
                    speed: 800,
                    animateGradually: {
                        enabled: true,
                        delay: 150
                    },
                    dynamicAnimation: {
                        enabled: true,
                        speed: 350
                    }
                }
            },
            colors: ['#4361ee'],
            dataLabels: {
                enabled: false
            },
            stroke: {
                curve: 'smooth',
                width: 3
            },
            fill: {
                type: 'gradient',
                gradient: {
                    shadeIntensity: 1,
                    opacityFrom: 0.7,
                    opacityTo: 0.2,
                    stops: [0, 90, 100]
                }
            },
            xaxis: {
                categories: ['Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set', 'Out', 'Nov', 'Dez']
            },
            tooltip: {
                theme: 'dark',
                y: {
                    formatter: function(val) {
                        return 'R$ ' + formatMoney(val);
                    }
                },
                custom: function({ series, seriesIndex, dataPointIndex, w }) {
                    const mes = w.globals.categoryLabels[dataPointIndex];
                    const valor = series[seriesIndex][dataPointIndex];
                    // Estimativa de quantidade: Vamos assumir que cada venda tem em média R$ 150
                    // Se você tiver o valor real no backend, substitua essa lógica
                    const qtdEstimada = Math.round(valor / 150);
                    return `<div class="custom-tooltip">
                        <div class="tooltip-title">${mes}</div>
                        <div class="tooltip-value">Total: R$ ${formatMoney(valor)}</div>
                        <div class="tooltip-qty">Quantidade est.: ${qtdEstimada} vendas</div>
                    </div>`;
                }
            },
            grid: {
                borderColor: 'rgba(163, 174, 208, 0.1)',
                strokeDashArray: 4,
                yaxis: {
                    lines: {
                        show: true
                    }
                }
            }
        };

        const salesChart = new ApexCharts(document.getElementById('salesChart'), salesChartOptions);
        salesChart.render();
        window.salesChart = salesChart;
    }

    if (document.getElementById('revenueChart')) {
        const chartElement = document.getElementById('revenueChart');
        let revenueData = [0, 0, 0, 0, 0, 0, 0]; // Valores padrão
        
        // Verificar se há dados reais disponíveis
        if (chartElement.dataset.revenue) {
            try {
                // Garantir que os dados sejam um array válido
                const parsedData = JSON.parse(chartElement.dataset.revenue);
                if (Array.isArray(parsedData)) {
                    revenueData = parsedData;
                } else if (typeof parsedData === 'object') {
                    // Se for um objeto, extrair os valores em um array
                    revenueData = Object.values(parsedData);
                }
            } catch (e) {
                console.error('Erro ao parsear dados de receita:', e);
            }
        }
        
        const revenueChartOptions = {
            series: [{
                name: 'Receitas',
                data: revenueData
            }],
            chart: {
                height: 350,
                type: 'bar',
                fontFamily: 'inherit',
                toolbar: {
                    show: false
                },
                animations: {
                    enabled: true,
                    easing: 'easeinout',
                    speed: 800
                }
            },
            plotOptions: {
                bar: {
                    borderRadius: 10,
                    columnWidth: '60%',
                    distributed: false,
                    endingShape: 'rounded'
                }
            },
            colors: ['#4D76E3', '#5A84EE', '#6693F8', '#73A1FF', '#80B0FF'],
            dataLabels: {
                enabled: false
            },
            fill: {
                type: 'gradient',
                gradient: {
                    shade: 'light',
                    type: 'vertical',
                    opacityFrom: 0.85,
                    opacityTo: 0.55,
                    stops: [0, 100]
                }
            },
            xaxis: {
                categories: ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta', 'Sábado', 'Domingo']
            },
            tooltip: {
                theme: 'dark',
                y: {
                    formatter: function(val) {
                        return 'R$ ' + formatMoney(val);
                    }
                },
                custom: function({ series, seriesIndex, dataPointIndex, w }) {
                    const dia = w.globals.categoryLabels[dataPointIndex];
                    const valor = series[seriesIndex][dataPointIndex];
                    // Estimativa de quantidade: Vamos assumir que cada compra tem em média R$ 300
                    // Se você tiver o valor real no backend, substitua essa lógica
                    const qtdEstimada = Math.round(valor / 300);
                    return `<div class="custom-tooltip">
                        <div class="tooltip-title">${dia}</div>
                        <div class="tooltip-value">Total: R$ ${formatMoney(valor)}</div>
                        <div class="tooltip-qty">Quantidade est.: ${qtdEstimada} compras</div>
                    </div>`;
                }
            },
            grid: {
                borderColor: 'rgba(163, 174, 208, 0.1)',
                strokeDashArray: 4
            }
        };

        const revenueChart = new ApexCharts(document.getElementById('revenueChart'), revenueChartOptions);
        revenueChart.render();
        window.revenueChart = revenueChart;
    }

    if (document.getElementById('categoriesChart')) {
        const chartElement = document.getElementById('categoriesChart');
        let categoriesValues = [0, 0, 0, 0, 0]; // Valores padrão
        let categoriesLabels = ['Categoria 1', 'Categoria 2', 'Categoria 3', 'Categoria 4', 'Categoria 5']; // Labels padrão
        
        // Verificar se há dados reais disponíveis
        if (chartElement.dataset.values && chartElement.dataset.labels) {
            try {
                // Garantir que os dados sejam arrays válidos
                const parsedValues = JSON.parse(chartElement.dataset.values);
                const parsedLabels = JSON.parse(chartElement.dataset.labels);
                
                if (Array.isArray(parsedValues)) {
                    categoriesValues = parsedValues;
                } else if (typeof parsedValues === 'object') {
                    categoriesValues = Object.values(parsedValues);
                }
                
                if (Array.isArray(parsedLabels)) {
                    categoriesLabels = parsedLabels;
                } else if (typeof parsedLabels === 'object') {
                    categoriesLabels = Object.values(parsedLabels);
                }
            } catch (e) {
                console.error('Erro ao parsear dados de categorias:', e);
            }
        }
        
        const categoriesChartOptions = {
            series: categoriesValues,
            chart: {
                height: 350,
                type: 'donut',
                fontFamily: 'inherit'
            },
            labels: categoriesLabels,
            colors: ['#4361ee', '#25d5e4', '#ff6a00', '#0AB39C', '#8251DC'],
            plotOptions: {
                pie: {
                    donut: {
                        size: '70%',
                        background: 'transparent',
                        labels: {
                            show: true,
                            name: {
                                show: true,
                                fontSize: '22px',
                                fontWeight: 600
                            },
                            value: {
                                show: true,
                                fontSize: '16px',
                                fontWeight: 400,
                                formatter: function(val) {
                                    return val;
                                }
                            },
                            total: {
                                show: true,
                                label: 'Total',
                                formatter: function() {
                                    return categoriesValues.reduce((a, b) => a + b, 0);
                                }
                            }
                        }
                    }
                }
            },
            dataLabels: {
                enabled: false
            },
            legend: {
                position: 'bottom',
                horizontalAlign: 'center',
                fontSize: '14px',
                markers: {
                    width: 10,
                    height: 10,
                    radius: 100
                },
                itemMargin: {
                    horizontal: 8,
                    vertical: 8
                }
            },
            stroke: {
                width: 0
            },
            fill: {
                type: 'gradient'
            }
        };

        const categoriesChart = new ApexCharts(document.getElementById('categoriesChart'), categoriesChartOptions);
        categoriesChart.render();
        window.categoriesChart = categoriesChart;
    }
}

/**
 * Configura os filtros de data
 */
function setupDateFilters() {
    const dateButtons = document.querySelectorAll('.date-range-btn');
    const customDateRange = document.getElementById('customDateRange');
    
    if (dateButtons.length && customDateRange) {
        dateButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Remove a classe ativa de todos os botões
                dateButtons.forEach(btn => btn.classList.remove('active'));
                // Adiciona a classe ativa ao botão clicado
                this.classList.add('active');
                
                const range = this.getAttribute('data-range');
                console.log('Botão de período clicado:', range);
                
                // Atualizar a exibição do período
                updatePeriodDisplay(range);
                
                // Mostra o campo personalizado se o botão "Personalizado" for clicado
                if (range === 'custom') {
                    customDateRange.classList.remove('hidden');
                    customDateRange.classList.add('visible');
                    
                    // Inicializar o flatpickr (seletor de data)
                    if (!customDateRange.flatpickr) {
                        flatpickr(customDateRange, {
                            mode: 'range',
                            dateFormat: 'd/m/Y',
                            locale: 'pt',
                            onClose: function(selectedDates, dateStr) {
                                console.log('Período personalizado selecionado:', dateStr);
                                // Atualizar o valor do campo escondido
                                if (document.getElementById('inp-periodo')) {
                                    document.getElementById('inp-periodo').value = dateStr;
                                }
                                updateCardFootersByPeriod(dateStr);
                                updatePeriodDisplay(dateStr);
                                updateDashboardData(dateStr);
                            }
                        });
                    }
                } else {
                    customDateRange.classList.remove('visible');
                    customDateRange.classList.add('hidden');
                    
                    // Atualizar o valor do campo escondido
                    if (document.getElementById('inp-periodo')) {
                        document.getElementById('inp-periodo').value = range;
                    }
                    
                    updateDashboardData(range);
                }
            });
        });
        
        // Se já tiver um período personalizado selecionado, inicializar o flatpickr
        if (document.querySelector('.date-range-btn[data-range="custom"].active') && !customDateRange.classList.contains('hidden')) {
            flatpickr(customDateRange, {
                mode: 'range',
                dateFormat: 'd/m/Y',
                locale: 'pt',
                onClose: function(selectedDates, dateStr) {
                    console.log('Período personalizado atualizado:', dateStr);
                    // Atualizar o valor do campo escondido
                    if (document.getElementById('inp-periodo')) {
                        document.getElementById('inp-periodo').value = dateStr;
                    }
                    updateCardFootersByPeriod(dateStr);
                    updatePeriodDisplay(dateStr);
                    updateDashboardData(dateStr);
                }
            });
        }
        
        // Atualizar a exibição do período inicial
        const initialPeriod = document.getElementById('inp-periodo') ? 
            document.getElementById('inp-periodo').value : 'month';
        updatePeriodDisplay(initialPeriod);
    }
}

/**
 * Atualiza os dados do dashboard com base no filtro selecionado
 * 
 * Esta função faz uma requisição AJAX ao servidor para obter dados reais
 * do banco de dados filtrados pelo período selecionado.
 * Não há geração de dados aleatórios - todos os valores vêm do backend.
 */
function updateDashboardData(range) {
    // Mostrar indicador de carregamento
    const dashboardContainer = document.querySelector('.dashboard-container');
    let loadingOverlay = null; // Declarando a variável no escopo da função
    
    // Atualizar exibição do período
    updatePeriodDisplay(range);
    
    if (dashboardContainer) {
        // Adiciona uma classe para efeito de carregamento
        dashboardContainer.classList.add('loading');
        
        // Adiciona spinner de carregamento
        loadingOverlay = document.createElement('div');
        loadingOverlay.className = 'dashboard-loading-overlay';
        loadingOverlay.innerHTML = '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Carregando...</span></div>';
        dashboardContainer.appendChild(loadingOverlay);
    }
    
    // Atualizar período ativo no filtro escondido
    if (document.getElementById('inp-periodo')) {
        document.getElementById('inp-periodo').value = range;
    } else {
        // Se não existir, cria o input oculto
        const hiddenInput = document.createElement('input');
        hiddenInput.type = 'hidden';
        hiddenInput.id = 'inp-periodo';
        hiddenInput.value = range;
        const dashboardFilters = document.querySelector('.dashboard-filters');
        if (dashboardFilters) {
            dashboardFilters.appendChild(hiddenInput);
        }
    }
    
    // Atualizar textos dos cards conforme período
    updateCardFootersByPeriod(range);
    
    // Verificar se path_url está definido, se não, definir um valor padrão
    let apiUrl;
    
    if (typeof path_url !== 'undefined' && path_url) {
        apiUrl = path_url;
    } else {
        apiUrl = window.location.origin + '/';
        console.log('path_url não definido. Usando', apiUrl);
    }
    
    // Verificar se a URL termina com uma barra
    if (apiUrl.substr(-1) !== '/') {
        apiUrl += '/';
    }

    // Parâmetros da requisição
    const params = {
        periodo: range
    };
    
    // Adicionar outros parâmetros apenas se estiverem disponíveis
    if ($('#empresa_id').length && $('#empresa_id').val()) {
        params.empresa_id = $('#empresa_id').val();
    }
    
    if ($('#usuario_id').length && $('#usuario_id').val()) {
        params.usuario_id = $('#usuario_id').val();
    }
    
    if ($('#inp-local_id').length && $('#inp-local_id').val()) {
        params.local_id = $('#inp-local_id').val();
    }
    
    console.log('Fazendo requisição para:', apiUrl + "api/graficos/dados-cards");
    console.log('Parâmetros:', params);
    
    // Fazer requisição AJAX para pegar os dados conforme o período
    $.get(apiUrl + "api/graficos/dados-cards", params)
    .done((response) => {
        console.log('Dados recebidos com sucesso:', response);
        
        // Processar a resposta - às vezes o backend pode retornar um objeto com propriedade 'data'
        let success;
        
        if (response && typeof response === 'object') {
            // Se a resposta tiver uma propriedade 'data', use-a
            if (response.data !== undefined) {
                success = response.data;
            } 
            // Caso contrário, a resposta inteira é o objeto de dados
            else {
                success = response;
            }
        } else {
            // Fallback se a resposta não for um objeto
            success = {};
            console.error('Formato de resposta inesperado:', response);
        }
        
        // Verificar se temos os dados do período solicitado
        if (success && success.periodo !== undefined && success.periodo !== range) {
            console.warn('Período retornado diferente do solicitado:', success.periodo, 'vs', range);
        }
        
        // Atualizar cards com os novos dados
        atualizarDadosDashboard(success);
        
        // Remover overlay de carregamento
        if (loadingOverlay) {
            loadingOverlay.remove();
        }
        if (dashboardContainer) {
            dashboardContainer.classList.remove('loading');
        }
        
        // Adicionar notificação de sucesso
        mostrarMensagemAtualizacao('Dados atualizados com sucesso!', 'success');
    })
    .fail((err) => {
        console.error('Erro ao atualizar dados:', err);
        
        // Remover overlay de carregamento
        if (loadingOverlay) {
            loadingOverlay.remove();
        }
        if (dashboardContainer) {
            dashboardContainer.classList.remove('loading');
        }
        
        // Mostrar mensagem de erro
        mostrarMensagemAtualizacao('Erro ao atualizar dados. Tente novamente.', 'error');
    });
}

/**
 * Atualiza os textos dos footers dos cards com base no período selecionado
 */
function updateCardFootersByPeriod(period) {
    // Obter os footers dos cards
    const receberFooter = document.querySelector('.diagonal-gradient-card.primary .card-footer span');
    const pagarFooter = document.querySelector('.diagonal-gradient-card.danger .card-footer span');
    
    if (!receberFooter || !pagarFooter) return;
    
    // Definir textos conforme período
    let footerText = 'Próximos 30 dias';
    
    switch (period) {
        case 'today':
            footerText = 'Vencimento hoje';
            break;
        case 'week':
            footerText = 'Próximos 7 dias';
            break;
        case 'month':
            footerText = 'Este mês';
            break;
        case 'year':
            footerText = 'Este ano';
            break;
        case 'custom':
            // Para período personalizado, vamos verificar se temos a data selecionada
            const customDateRange = document.getElementById('customDateRange');
            if (customDateRange && customDateRange.value) {
                footerText = 'Período: ' + customDateRange.value;
            } else {
                footerText = 'Período personalizado';
            }
            break;
        default:
            // Se o período for uma data específica (ex: custom range)
            if (period.includes('/')) {
                footerText = 'Período: ' + period;
            } else {
                footerText = 'Próximos 30 dias';
            }
    }
    
    // Atualizar textos
    receberFooter.textContent = footerText;
    pagarFooter.textContent = footerText;
}

/**
 * Atualiza todos os elementos do dashboard com os novos dados
 */
function atualizarDadosDashboard(dados) {
    console.log('Dados recebidos para atualização:', dados); // Log para diagnóstico
    
    // Verificar o período atual
    const periodoAtual = document.getElementById('inp-periodo') ? 
        document.getElementById('inp-periodo').value : 'today';
    console.log('Período atual:', periodoAtual);
    
    // Atualizar a exibição do período com as datas exatas da API, se disponíveis
    if (dados['data_inicio'] && dados['data_fim']) {
        // Converter formato YYYY-MM-DD para DD/MM/YYYY para display
        const dataInicio = dados['data_inicio'].split('-').reverse().join('/');
        const dataFim = dados['data_fim'].split('-').reverse().join('/');
        
        // Se início e fim forem iguais, mostrar apenas uma data
        const periodoFormatado = dataInicio === dataFim ? 
            dataInicio : 
            dataInicio + ' - ' + dataFim;
            
        // Atualizar a exibição do período com os dados exatos da API
        const periodDateRangeEl = document.getElementById('period-date-range');
        if (periodDateRangeEl) {
            periodDateRangeEl.textContent = '(' + periodoFormatado + ')';
        }
    }
    
    // Atualizar cards principais com valores baseados no período
    if (dados['clientes'] !== undefined) {
        $('.total-clientes').text(dados['clientes'] || '0');
    }
    
    if (dados['produtos'] !== undefined) {
        $('.total-produtos').text(dados['produtos'] || '0');
    }
    
    // Verificar se há valores específicos para o período
    if (dados['vendas_periodo'] !== undefined) {
        $('.total-vendas').text("R$ " + convertFloatToMoeda(dados['vendas_periodo'] || 0));
        console.log('Usando vendas_periodo:', dados['vendas_periodo']);
    } else {
        $('.total-vendas').text("R$ " + convertFloatToMoeda(dados['vendas'] || 0));
        console.log('Usando vendas padrão:', dados['vendas']);
    }
    
    if (dados['compras_periodo'] !== undefined) {
        $('.total-compras').text("R$ " + convertFloatToMoeda(dados['compras_periodo'] || 0));
        console.log('Usando compras_periodo:', dados['compras_periodo']);
    } else {
        $('.total-compras').text("R$ " + convertFloatToMoeda(dados['compras'] || 0));
        console.log('Usando compras padrão:', dados['compras']);
    }
    
    // Atualizar contas a receber e pagar conforme o período
    if (dados['contas_receber_periodo'] !== undefined) {
        $('.total-receber').text("R$ " + convertFloatToMoeda(dados['contas_receber_periodo'] || 0));
        console.log('Usando contas_receber_periodo:', dados['contas_receber_periodo']);
    } else {
        $('.total-receber').text("R$ " + convertFloatToMoeda(dados['contas_receber'] || 0));
        console.log('Usando contas_receber padrão:', dados['contas_receber']);
    }
    
    if (dados['contas_pagar_periodo'] !== undefined) {
        $('.total-pagar').text("R$ " + convertFloatToMoeda(dados['contas_pagar_periodo'] || 0));
        console.log('Usando contas_pagar_periodo:', dados['contas_pagar_periodo']);
    } else {
        $('.total-pagar').text("R$ " + convertFloatToMoeda(dados['contas_pagar'] || 0));
        console.log('Usando contas_pagar padrão:', dados['contas_pagar']);
    }
    
    // Atualizar estatísticas do cabeçalho se disponíveis
    // Verificar campos específicos do período
    if (dados['totalNfeCount_periodo'] !== undefined) {
        $('.dashboard-stat-item:nth-child(1) .stat-value').text(dados['totalNfeCount_periodo']);
    } else if (dados['totalNfeCount'] !== undefined) {
        $('.dashboard-stat-item:nth-child(1) .stat-value').text(dados['totalNfeCount']);
    }
    
    if (dados['totalNfceCount_periodo'] !== undefined) {
        $('.dashboard-stat-item:nth-child(2) .stat-value').text(dados['totalNfceCount_periodo']);
    } else if (dados['totalNfceCount'] !== undefined) {
        $('.dashboard-stat-item:nth-child(2) .stat-value').text(dados['totalNfceCount']);
    }
    
    if (dados['totalVendasMes_periodo'] !== undefined) {
        $('.dashboard-stat-item:nth-child(3) .stat-value').text("R$ " + convertFloatToMoeda(dados['totalVendasMes_periodo']));
    } else if (dados['totalVendasMes'] !== undefined) {
        $('.dashboard-stat-item:nth-child(3) .stat-value').text("R$ " + convertFloatToMoeda(dados['totalVendasMes']));
    }
    
    if (dados['totalComprasMes_periodo'] !== undefined) {
        $('.dashboard-stat-item:nth-child(4) .stat-value').text("R$ " + convertFloatToMoeda(dados['totalComprasMes_periodo']));
    } else if (dados['totalComprasMes'] !== undefined) {
        $('.dashboard-stat-item:nth-child(4) .stat-value').text("R$ " + convertFloatToMoeda(dados['totalComprasMes']));
    }
    
    // Atualizar gráficos se os dados estiverem disponíveis
    if (dados['somaVendasMesesAnteriores_periodo'] && window.salesChart) {
        let vendas = dados['somaVendasMesesAnteriores_periodo'];
        atualizarGraficoVendas(vendas);
    } else if (dados['somaVendasMesesAnteriores'] && window.salesChart) {
        let vendas = dados['somaVendasMesesAnteriores'];
        atualizarGraficoVendas(vendas);
    }
    
    if (dados['somaComprasMesesAnteriores_periodo'] && window.revenueChart) {
        let compras = dados['somaComprasMesesAnteriores_periodo'];
        atualizarGraficoCompras(compras);
    } else if (dados['somaComprasMesesAnteriores'] && window.revenueChart) {
        let compras = dados['somaComprasMesesAnteriores'];
        atualizarGraficoCompras(compras);
    }
    
    // Atualizar cards detalhados
    if (dados['totalVendasMes_periodo'] !== undefined) {
        $('.stat-card-ultramodern.primary .stat-value').text("R$ " + convertFloatToMoeda(dados['totalVendasMes_periodo']));
    } else if (dados['totalVendasMes'] !== undefined) {
        $('.stat-card-ultramodern.primary .stat-value').text("R$ " + convertFloatToMoeda(dados['totalVendasMes']));
    }
    
    if (dados['totalComprasMes_periodo'] !== undefined) {
        $('.stat-card-ultramodern.success .stat-value').text("R$ " + convertFloatToMoeda(dados['totalComprasMes_periodo']));
    } else if (dados['totalComprasMes'] !== undefined) {
        $('.stat-card-ultramodern.success .stat-value').text("R$ " + convertFloatToMoeda(dados['totalComprasMes']));
    }
    
    if (dados['totalEmitidoMes_periodo'] !== undefined) {
        $('.stat-card-ultramodern.info .stat-value').text("R$ " + convertFloatToMoeda(dados['totalEmitidoMes_periodo']));
    } else if (dados['totalEmitidoMes'] !== undefined) {
        $('.stat-card-ultramodern.info .stat-value').text("R$ " + convertFloatToMoeda(dados['totalEmitidoMes']));
    }
    
    // Atualizar distribuição de vendas se disponível
    atualizarGraficoDistribuicao(dados);
}

/**
 * Atualiza o gráfico de vendas
 */
function atualizarGraficoVendas(vendas) {
    // Garantir que os dados são um array válido
    if (!Array.isArray(vendas)) {
        if (typeof vendas === 'object') {
            vendas = Object.values(vendas);
        } else {
            vendas = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]; // Array padrão com 12 valores (meses)
        }
    }
    
    window.salesChart.updateSeries([{
        name: 'Vendas',
        data: vendas
    }]);
}

/**
 * Atualiza o gráfico de compras
 */
function atualizarGraficoCompras(compras) {
    // Garantir que os dados são um array válido
    if (!Array.isArray(compras)) {
        if (typeof compras === 'object') {
            compras = Object.values(compras);
        } else {
            compras = [0, 0, 0, 0, 0, 0, 0]; // Array padrão com 7 valores (dias da semana)
        }
    }
    
    window.revenueChart.updateSeries([{
        name: 'Compras',
        data: compras
    }]);
}

/**
 * Atualiza o gráfico de distribuição
 */
function atualizarGraficoDistribuicao(dados) {
    // Verificar campos específicos do período primeiro
    const temValoresEspecificos = 
        dados['totalNfeCount_periodo'] !== undefined || 
        dados['totalNfceCount_periodo'] !== undefined || 
        dados['totalCteCount_periodo'] !== undefined || 
        dados['totalMdfeCount_periodo'] !== undefined;
    
    if (temValoresEspecificos && window.categoriesChart) {
        const values = [
            dados['totalNfeCount_periodo'] || 0,
            dados['totalNfceCount_periodo'] || 0,
            dados['totalCteCount_periodo'] || 0,
            dados['totalMdfeCount_periodo'] || 0
        ];
        
        window.categoriesChart.updateOptions({
            series: values,
            labels: ["NFe", "NFCe", "CTe", "MDFe"]
        });
    } 
    // Se não tiver valores específicos, usar os valores padrão
    else if ((dados['totalNfeCount'] !== undefined || 
              dados['totalNfceCount'] !== undefined || 
              dados['totalCteCount'] !== undefined || 
              dados['totalMdfeCount'] !== undefined) && 
             window.categoriesChart) {
        
        const values = [
            dados['totalNfeCount'] || 0,
            dados['totalNfceCount'] || 0,
            dados['totalCteCount'] || 0,
            dados['totalMdfeCount'] || 0
        ];
        
        window.categoriesChart.updateOptions({
            series: values,
            labels: ["NFe", "NFCe", "CTe", "MDFe"]
        });
    }
}

/**
 * Mostra uma mensagem de atualização temporária
 */
function mostrarMensagemAtualizacao(mensagem, tipo) {
    // Criar elemento de notificação
    const notificacao = document.createElement('div');
    notificacao.className = 'dashboard-notification-toast ' + tipo;
    notificacao.innerHTML = `
        <div class="dashboard-notification-toast-content">
            <i class="ri-${tipo === 'success' ? 'check-double-line' : 'error-warning-line'}"></i>
            <span>${mensagem}</span>
        </div>
    `;
    
    // Adicionar ao corpo do documento
    document.body.appendChild(notificacao);
    
    // Mostrar com animação
    setTimeout(() => {
        notificacao.classList.add('show');
    }, 10);
    
    // Remover após 3 segundos
    setTimeout(() => {
        notificacao.classList.remove('show');
        setTimeout(() => {
            notificacao.remove();
        }, 300);
    }, 3000);
}

/**
 * Configura as animações ao scroll
 */
function setupScrollAnimations() {
    const animatedElements = document.querySelectorAll('.animate-on-scroll');
    
    // Função para verificar se um elemento está visível
    function isElementInViewport(el) {
        const rect = el.getBoundingClientRect();
        return (
            rect.top <= (window.innerHeight || document.documentElement.clientHeight) * 0.9 &&
            rect.bottom >= 0
        );
    }
    
    // Função para animar elementos visíveis
    function animateElementsInViewport() {
        animatedElements.forEach(element => {
            if (isElementInViewport(element) && !element.classList.contains('animated')) {
                element.classList.add('animated');
                
                // Aplica a classe de animação
                if (element.classList.contains('animate-fade-up')) {
                    element.style.animation = 'fadeInUp 0.5s ease-out forwards';
                } else if (element.classList.contains('animate-fade-left')) {
                    element.style.animation = 'fadeInLeft 0.5s ease-out forwards';
                } else if (element.classList.contains('animate-fade-right')) {
                    element.style.animation = 'fadeInRight 0.5s ease-out forwards';
                }
            }
        });
    }
    
    // Verifica na carga inicial e ao scroll
    document.addEventListener('scroll', animateElementsInViewport);
    window.addEventListener('resize', animateElementsInViewport);
    animateElementsInViewport(); // Executa na carga inicial
}

/**
 * Inicializa componentes Bootstrap
 */
function initializeBootstrapComponents() {
    // Ativa os tooltips
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });
    
    // Ativa os popovers
    const popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });
}

/**
 * Configura o layout responsivo
 */
function setupResponsiveLayout() {
    function adjustLayout() {
        const width = window.innerWidth;
        const dashboardQuarters = document.querySelectorAll('.dashboard-quarter');
        const dashboardThirds = document.querySelectorAll('.dashboard-third');
        
        if (width < 768) { // Mobile
            dashboardQuarters.forEach(item => {
                item.style.gridColumn = 'span 12';
            });
            dashboardThirds.forEach(item => {
                item.style.gridColumn = 'span 12';
            });
            document.querySelectorAll('.dashboard-half').forEach(item => {
                item.style.gridColumn = 'span 12';
            });
        } else if (width < 992) { // Tablet
            dashboardQuarters.forEach(item => {
                item.style.gridColumn = 'span 6';
            });
            dashboardThirds.forEach(item => {
                item.style.gridColumn = 'span 6';
            });
            document.querySelectorAll('.dashboard-half').forEach(item => {
                item.style.gridColumn = 'span 12';
            });
        } else { // Desktop
            dashboardQuarters.forEach(item => {
                item.style.gridColumn = 'span 3';
            });
            dashboardThirds.forEach(item => {
                item.style.gridColumn = 'span 4';
            });
            document.querySelectorAll('.dashboard-half').forEach(item => {
                item.style.gridColumn = 'span 6';
            });
        }
    }
    
    window.addEventListener('resize', adjustLayout);
    adjustLayout(); // Executa na carga inicial
}

/**
 * Atualiza os dados de um card específico
 * 
 * Esta função faz uma requisição AJAX para buscar dados reais atualizados
 * para o card específico. Os dados vêm diretamente do banco de dados.
 */
function refreshCardData(card) {
    const spinner = document.createElement('div');
    spinner.className = 'card-spinner';
    spinner.innerHTML = '<div class="spinner-border text-primary" role="status"><span class="visually-hidden">Carregando...</span></div>';
    
    // Adiciona o spinner ao card
    card.appendChild(spinner);
    card.classList.add('card-refreshing');
    
    // Obtém o ID ou tipo do card para saber o que atualizar
    const cardId = card.id || '';
    const cardType = card.getAttribute('data-card-type') || '';
    
    // URL da requisição AJAX (incluir ID ou tipo na requisição)
    const url = window.location.pathname + '?card=' + (cardId || cardType);
    
    // Fazer a requisição AJAX
    fetch(url)
        .then(response => response.text())
        .then(html => {
            // Criar um DOM temporário para extrair os dados
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            
            // Atualizar os valores no card
            if (cardId) {
                const newCard = doc.getElementById(cardId);
                if (newCard) {
                    // Atualizar estatísticas
                    const statValues = card.querySelectorAll('.stat-value, .card-value');
                    statValues.forEach(value => {
                        const newValue = newCard.querySelector('.' + value.className.split(' ')[0]);
                        if (newValue) {
                            value.textContent = newValue.textContent;
                        }
                    });
                    
                    // Atualizar qualquer gráfico presente
                    const chartElement = card.querySelector('[data-sparkline], [id$="Chart"]');
                    if (chartElement) {
                        const chartId = chartElement.id;
                        if (chartId && window[chartId]) {
                            // Se for um gráfico ApexCharts
                            const newChartData = newCard.querySelector('#' + chartId).dataset;
                            if (newChartData) {
                                if (newChartData.revenue) {
                                    window[chartId].updateSeries([{
                                        data: JSON.parse(newChartData.revenue)
                                    }]);
                                } else if (newChartData.sales) {
                                    window[chartId].updateSeries([{
                                        data: JSON.parse(newChartData.sales)
                                    }]);
                                } else if (newChartData.values && newChartData.labels) {
                                    window[chartId].updateOptions({
                                        series: JSON.parse(newChartData.values),
                                        labels: JSON.parse(newChartData.labels)
                                    });
                                }
                            }
                        } else if (chartElement.hasAttribute('data-sparkline')) {
                            // Se for um sparkline
                            const newSparkline = newCard.querySelector('[data-sparkline]');
                            if (newSparkline) {
                                const values = newSparkline.getAttribute('data-sparkline').split(',').map(v => parseInt(v));
                                // Recriar o sparkline
                                chartElement.innerHTML = '';
                                createSparkline(chartElement, values);
                            }
                        }
                    }
                }
            }
            
            // Remover o spinner
            spinner.remove();
            card.classList.remove('card-refreshing');
            
            // Mostrar animação de sucesso
            const successIcon = document.createElement('div');
            successIcon.className = 'refresh-success';
            successIcon.innerHTML = '<i class="ri-check-line"></i>';
            card.appendChild(successIcon);
            
            // Remover o ícone de sucesso após 1 segundo
            setTimeout(() => {
                successIcon.remove();
            }, 1000);
        })
        .catch(error => {
            console.error('Erro ao atualizar card:', error);
            
            // Remover o spinner
            spinner.remove();
            card.classList.remove('card-refreshing');
            
            // Mostrar animação de erro
            const errorIcon = document.createElement('div');
            errorIcon.className = 'refresh-error';
            errorIcon.innerHTML = '<i class="ri-error-warning-line"></i>';
            card.appendChild(errorIcon);
            
            // Remover o ícone de erro após 1 segundo
            setTimeout(() => {
                errorIcon.remove();
            }, 1000);
        });
}

function toggleFullscreen(card) {
    if (!card.classList.contains('fullscreen')) {
        card.classList.add('fullscreen');
        document.body.classList.add('card-fullscreen-active');
        
        // Cria um backdrop
        const backdrop = document.createElement('div');
        backdrop.className = 'card-fullscreen-backdrop';
        backdrop.addEventListener('click', () => {
            exitFullscreen(card);
        });
        document.body.appendChild(backdrop);
        
        // Cria um botão de fechar
        const closeBtn = document.createElement('button');
        closeBtn.className = 'card-fullscreen-close';
        closeBtn.innerHTML = '<i class="ri-close-line"></i>';
        closeBtn.addEventListener('click', () => {
            exitFullscreen(card);
        });
        card.appendChild(closeBtn);
        
        // Reajusta os gráficos (se existirem)
        setTimeout(() => {
            const chartContainer = card.querySelector('.chart-container');
            if (chartContainer) {
                const chartId = chartContainer.id;
                if (chartId === 'salesChart' && window.salesChart) {
                    window.salesChart.render();
                } else if (chartId === 'revenueChart' && window.revenueChart) {
                    window.revenueChart.render();
                } else if (chartId === 'categoriesChart' && window.categoriesChart) {
                    window.categoriesChart.render();
                }
            }
        }, 300);
    }
}

function exitFullscreen(card) {
    card.classList.remove('fullscreen');
    document.body.classList.remove('card-fullscreen-active');
    
    // Remove o backdrop
    const backdrop = document.querySelector('.card-fullscreen-backdrop');
    if (backdrop) {
        backdrop.remove();
    }
    
    // Remove o botão de fechar
    const closeBtn = card.querySelector('.card-fullscreen-close');
    if (closeBtn) {
        closeBtn.remove();
    }
    
    // Reajusta os gráficos (se existirem)
    setTimeout(() => {
        const chartContainer = card.querySelector('.chart-container');
        if (chartContainer) {
            const chartId = chartContainer.id;
            if (chartId === 'salesChart' && window.salesChart) {
                window.salesChart.render();
            } else if (chartId === 'revenueChart' && window.revenueChart) {
                window.revenueChart.render();
            } else if (chartId === 'categoriesChart' && window.categoriesChart) {
                window.categoriesChart.render();
            }
        }
    }, 300);
}

function showCardOptions(btn) {
    // Implementação de menu de opções para o card
    // Poderia ser um dropdown ou um modal com opções
    const card = btn.closest('.card');
    const cardTitle = card.querySelector('.card-header h5').textContent;
    
    // Cria um dropdown de opções
    const dropdown = document.createElement('div');
    dropdown.className = 'card-options-dropdown';
    dropdown.innerHTML = `
        <ul>
            <li data-action="refresh"><i class="ri-refresh-line"></i> Atualizar</li>
            <li data-action="download"><i class="ri-download-line"></i> Exportar</li>
            <li data-action="fullscreen"><i class="ri-fullscreen-line"></i> Expandir</li>
            <li data-action="settings"><i class="ri-settings-3-line"></i> Configurações</li>
        </ul>
    `;
    
    // Posiciona o dropdown
    const rect = btn.getBoundingClientRect();
    dropdown.style.top = rect.bottom + 'px';
    dropdown.style.right = (window.innerWidth - rect.right) + 'px';
    
    // Adiciona ao body
    document.body.appendChild(dropdown);
    
    // Adiciona eventos aos itens
    dropdown.querySelectorAll('li').forEach(item => {
        item.addEventListener('click', function() {
            const action = this.getAttribute('data-action');
            
            if (action === 'refresh') {
                refreshCardData(card);
            } else if (action === 'fullscreen') {
                toggleFullscreen(card);
            } else if (action === 'download') {
                // Implementar exportação
                alert(`Exportando dados de "${cardTitle}"`);
            } else if (action === 'settings') {
                // Implementar configurações
                alert(`Configurações para "${cardTitle}"`);
            }
            
            // Remove o dropdown
            dropdown.remove();
        });
    });
    
    // Fecha o dropdown ao clicar fora
    document.addEventListener('click', function closeDropdown(e) {
        if (!dropdown.contains(e.target) && e.target !== btn) {
            dropdown.remove();
            document.removeEventListener('click', closeDropdown);
        }
    });
}

/**
 * Funções auxiliares
 */

function formatMoney(value) {
    return new Intl.NumberFormat('pt-BR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    }).format(value);
}

/**
 * Função para converter valores para formato de moeda brasileira
 */
function convertFloatToMoeda(value) {
    if (!value) return '0,00';
    return new Intl.NumberFormat('pt-BR', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    }).format(value);
}

/**
 * Atualiza a exibição do período atual no cabeçalho
 */
function updatePeriodDisplay(period, startDate, endDate) {
    const periodDisplayEl = document.getElementById('current-period-display');
    const periodDateRangeEl = document.getElementById('period-date-range');
    
    if (!periodDisplayEl || !periodDateRangeEl) return;
    
    let periodText = '';
    let dateRangeText = '';
    
    // Definir o texto do período
    switch(period) {
        case 'today':
            periodText = 'Hoje';
            dateRangeText = '(' + formatDateBR(new Date()) + ')';
            break;
        case 'week':
            periodText = 'Esta Semana';
            const weekStart = getStartOfWeek(new Date());
            const weekEnd = getEndOfWeek(new Date());
            dateRangeText = '(' + formatDateBR(weekStart) + ' - ' + formatDateBR(weekEnd) + ')';
            break;
        case 'month':
            periodText = 'Este Mês';
            const monthStart = new Date(new Date().getFullYear(), new Date().getMonth(), 1);
            const monthEnd = new Date(new Date().getFullYear(), new Date().getMonth() + 1, 0);
            dateRangeText = '(' + formatDateBR(monthStart) + ' - ' + formatDateBR(monthEnd) + ')';
            break;
        case 'year':
            periodText = 'Este Ano';
            const yearStart = new Date(new Date().getFullYear(), 0, 1);
            const yearEnd = new Date(new Date().getFullYear(), 11, 31);
            dateRangeText = '(' + formatDateBR(yearStart) + ' - ' + formatDateBR(yearEnd) + ')';
            break;
        default:
            // Se for um período personalizado
            if (period.includes('/')) {
                periodText = 'Período Personalizado';
                dateRangeText = '(' + period + ')';
            } else {
                periodText = 'Todos os Dados';
                dateRangeText = '';
            }
    }
    
    // Atualizar elementos
    periodDisplayEl.textContent = periodText;
    periodDateRangeEl.textContent = dateRangeText;
}

/**
 * Formata data no padrão brasileiro (dd/mm/yyyy)
 */
function formatDateBR(date) {
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    return `${day}/${month}/${year}`;
}

/**
 * Obtém o início da semana atual
 */
function getStartOfWeek(date) {
    const dayOfWeek = date.getDay();
    const diff = date.getDate() - dayOfWeek + (dayOfWeek === 0 ? -6 : 1); // ajusta se o domingo for 0
    return new Date(date.setDate(diff));
}

/**
 * Obtém o fim da semana atual
 */
function getEndOfWeek(date) {
    const startOfWeek = getStartOfWeek(date);
    const endOfWeek = new Date(startOfWeek);
    endOfWeek.setDate(startOfWeek.getDate() + 6);
    return endOfWeek;
}

// Exportar funções para uso global
window.updateDashboardData = updateDashboardData;
window.atualizarDadosDashboard = atualizarDadosDashboard;
window.mostrarMensagemAtualizacao = mostrarMensagemAtualizacao;
window.convertFloatToMoeda = convertFloatToMoeda; 