<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\Cliente;
use App\Models\ContaPagar;
use App\Models\Fornecedor;
use App\Models\ItemContaEmpresa;
use App\Models\ManutencaoVeiculo;
use App\Models\DespesaFrete;
use App\Models\CategoriaConta;
use App\Models\CentroCusto;
use App\Models\AnexoConta;
use App\Models\AlocacaoCentroCusto;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Utils\ContaEmpresaUtil;
use App\Utils\UploadUtil;
use Carbon\Carbon;

class ContaPagarController extends Controller
{

    protected $util;
    protected $uploadUtil;
    public function __construct(ContaEmpresaUtil $util, UploadUtil $uploadUtil){
        $this->util = $util;
        $this->uploadUtil = $uploadUtil;
        $this->middleware('permission:conta_pagar_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:conta_pagar_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:conta_pagar_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:conta_pagar_delete', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {

        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);

        $fornecedor_id = $request->fornecedor_id;
        $start_date = $request->start_date;
        $end_date = $request->end_date;
        $status = $request->status;
        $ordem = $request->ordem;
        $local_id = $request->get('local_id');
        $categoria_id = $request->categoria_id;
        $prioridade = $request->prioridade;
        $recorrente = $request->recorrente;
        $vencidas = $request->vencidas;
        $parcialmente_pagas = $request->parcialmente_pagas;

        $data = ContaPagar::where('empresa_id', request()->empresa_id)
        ->when(!empty($fornecedor_id), function ($query) use ($fornecedor_id) {
            return $query->where('fornecedor_id', $fornecedor_id);
        })
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('data_vencimento', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date) {
            return $query->whereDate('data_vencimento', '<=', $end_date);
        })
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->when($status != '', function ($query) use ($status) {
            return $query->where('status', $status);
        })
        ->when($categoria_id, function ($query) use ($categoria_id) {
            return $query->where('categoria_id', $categoria_id);
        })
        ->when($prioridade, function ($query) use ($prioridade) {
            return $query->where('prioridade', $prioridade);
        })
        ->when($recorrente, function ($query) {
            return $query->where('recorrente', true);
        })
        ->when($vencidas, function ($query) {
            return $query->where('status', 0)
                         ->whereDate('data_vencimento', '<', now());
        })
        ->when($parcialmente_pagas, function ($query) {
            return $query->where('status', 0)
                         ->whereNotNull('valor_pago')
                         ->whereRaw('valor_pago > 0')
                         ->whereRaw('valor_pago < valor_integral');
        })
        ->when($ordem != '', function ($query) use ($ordem) {
            return $query->orderBy('data_vencimento', 'asc');
        })
        ->when($ordem == '', function ($query) use ($ordem) {
            return $query->orderBy('created_at', 'asc');
        })
        ->with(['fornecedor', 'categoria', 'localizacao'])
        ->paginate(env("PAGINACAO"));

        $fornecedor = null;
        if($fornecedor_id){
            $fornecedor = Cliente::findOrFail($fornecedor_id);
        }

        $categorias = CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'pagar')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();

        return view('conta-pagar.index', compact('data', 'fornecedor', 'categorias'));
    }

    public function create()
    {
        $fornecedores = Fornecedor::where('empresa_id', request()->empresa_id)->get();
        
        $categorias = CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'pagar')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $centrosCusto = CentroCusto::where('empresa_id', request()->empresa_id)
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $periodicidades = ContaPagar::periodicidades();
        $prioridades = ContaPagar::prioridades();
        
        return view('conta-pagar.create', compact('fornecedores', 'categorias', 'centrosCusto', 'periodicidades', 'prioridades'));
    }

    public function store(Request $request)
    {
        $this->__validate($request);
        try {
            DB::beginTransaction();

            $file_name = '';
            if ($request->hasFile('file')) {
                $file_name = $this->uploadUtil->uploadFile($request->file, '/financeiro');
            }

            // Definir nível de aprovação baseado no valor
            $valor_integral = __convert_value_bd($request->valor_integral);
            $temp = new ContaPagar();
            $temp->valor_integral = $valor_integral;
            $nivel_aprovacao = $temp->defineNivelAprovacao();
            
            // Tratar dados básicos da conta
            $request->merge([
                'valor_integral' => $valor_integral,
                'valor_pago' => $request->valor_pago ? __convert_value_bd($request->valor_pago) : 0,
                'arquivo' => $file_name,
                'nivel_aprovacao' => $nivel_aprovacao,
                'recorrente' => $request->has('recorrente'),
            ]);
            
            // Auto-aprovar contas abaixo de determinado valor
            if ($valor_integral <= 1000) {
                $request->merge([
                    'aprovado_por' => Auth::id(),
                    'data_aprovacao' => now(),
                ]);
            }
            
            // Criar conta principal
            $conta = ContaPagar::create($request->all());
            
            // Salvar alocações de centro de custo se fornecidas
            if ($request->has('centro_custo_valores') && is_array($request->centro_custo_valores)) {
                foreach ($request->centro_custo_valores as $index => $valor) {
                    if (!empty($valor)) {
                        $valorAlocacao = __convert_value_bd($valor);
                        $porcentagem = ($valorAlocacao / $valor_integral) * 100;
                        
                        AlocacaoCentroCusto::create([
                            'conta_pagar_id' => $conta->id,
                            'centro_custo_id' => $request->centro_custo_ids[$index],
                            'valor' => $valorAlocacao,
                            'porcentagem' => $porcentagem,
                            'observacao' => $request->centro_custo_observacoes[$index] ?? null
                        ]);
                    }
                }
            }
            
            // Processar múltiplos anexos
            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $anexo) {
                    $nomeArquivo = uniqid() . '.' . $anexo->getClientOriginalExtension();
                    $caminhoArquivo = $anexo->storeAs('financeiro/anexos', $nomeArquivo, 'public');
                    
                    AnexoConta::create([
                        'conta_pagar_id' => $conta->id,
                        'nome_arquivo' => $anexo->getClientOriginalName(),
                        'caminho_arquivo' => $caminhoArquivo,
                        'tipo_arquivo' => $anexo->getClientMimeType(),
                        'tamanho_arquivo' => $anexo->getSize(),
                        'usuario_id' => Auth::id()
                    ]);
                }
            }

            $descricaoLog = "Vencimento: " . __data_pt($request->data_vencimento, 0) . " R$ " . __moeda($request->valor_integral);
            __createLog(request()->empresa_id, 'Conta a Pagar', 'cadastrar', $descricaoLog);
            
            // Vincular a manutenção de veículo, se necessário
            if(isset($request->manutencao_id)){
                $manutencao = ManutencaoVeiculo::findOrFail($request->manutencao_id);
                $manutencao->conta_pagar_id = $conta->id;
                $manutencao->save();
            }

            // Vincular a despesa de frete, se necessário
            if(isset($request->despesa_id)){
                $despesa = DespesaFrete::findOrFail($request->despesa_id);
                $despesa->conta_pagar_id = $conta->id;
                $despesa->save();
            }

            // Gerar parcelas recorrentes, se necessário
            if ($request->recorrente && $request->periodicidade && $request->total_parcelas > 0) {
                $conta->periodicidade = $request->periodicidade;
                $conta->total_parcelas = $request->total_parcelas;
                $conta->parcela_atual = 1;
                $conta->save();
                
                // Gerar parcelas futuras
                for ($i = 2; $i <= $request->total_parcelas; $i++) {
                    $parcela = clone $conta;
                    $parcela->status = 0;
                    $parcela->valor_pago = 0;
                    $parcela->data_pagamento = null;
                    $parcela->aprovado_por = null;
                    $parcela->data_aprovacao = null;
                    $parcela->parent_id = $conta->id;
                    $parcela->parcela_atual = $i;
                    
                    // Garantir que a descrição seja mantida
                    $parcela->descricao = $conta->descricao;
                    
                    // Calcular próxima data de vencimento baseado na periodicidade
                    $dataBase = $conta->data_vencimento;
                    switch ($request->periodicidade) {
                        case ContaPagar::PERIODICIDADE_DIARIA:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' days'));
                            break;
                        case ContaPagar::PERIODICIDADE_SEMANAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' weeks'));
                            break;
                        case ContaPagar::PERIODICIDADE_QUINZENAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 15) . ' days'));
                            break;
                        case ContaPagar::PERIODICIDADE_MENSAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_BIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 2) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_TRIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 3) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_SEMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 6) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_ANUAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' years'));
                            break;
                        default:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                    }
                    
                    $parcela->data_vencimento = $dataVencimento;
                    $parcela->save();
                    
                    $descricaoLog = "Vencimento (parcela $i): " . __data_pt($dataVencimento, 0) . " R$ " . __moeda($conta->valor_integral);
                    __createLog(request()->empresa_id, 'Conta a Pagar', 'cadastrar', $descricaoLog);
                }
            }
            
            // Processar recorrência antiga se houver datas específicas
            elseif ($request->dt_recorrencia) {
                for ($i = 0; $i < sizeof($request->dt_recorrencia); $i++) {
                    $data = $request->dt_recorrencia[$i];
                    $valor = __convert_value_bd($request->valor_recorrencia[$i]);
                    $data = [
                        'venda_id' => null,
                        'data_vencimento' => $data,
                        'data_pagamento' => $data,
                        'valor_integral' => $valor,
                        'valor_pago' => $request->status ? $valor : 0,
                        'referencia' => $request->referencia,
                        'categoria_id' => $request->categoria_id,
                        'status' => $request->status,
                        'empresa_id' => $request->empresa_id,
                        'fornecedor_id' => $request->fornecedor_id,
                        'tipo_pagamento' => $request->tipo_pagamento,
                        'local_id' => $conta->local_id,
                        'descricao' => $request->descricao // Garantir que a descrição seja mantida
                    ];

                    ContaPagar::create($data);

                    $descricaoLog = "Vencimento: " . __data_pt($request->dt_recorrencia[$i], 0) . " R$ " . __moeda($valor);
                    __createLog(request()->empresa_id, 'Conta a Pagar', 'cadastrar', $descricaoLog);
                }
            }
            
            DB::commit();
            session()->flash("flash_success", "Conta a Pagar cadastrada!");
        } catch (\Exception $e) {
            DB::rollBack();
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        if(isset($request->redirect)){
            return redirect($request->redirect);
        }
        return redirect()->route('conta-pagar.index');
    }

    public function edit($id)
    {
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);

        $fornecedores = Fornecedor::where('empresa_id', request()->empresa_id)->get();
        
        $categorias = CategoriaConta::where('empresa_id', request()->empresa_id)
            ->where(function($query) {
                $query->where('tipo', 'pagar')
                      ->orWhere('tipo', 'ambos');
            })
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $centrosCusto = CentroCusto::where('empresa_id', request()->empresa_id)
            ->where('ativo', true)
            ->orderBy('nome')
            ->get();
            
        $alocacoes = AlocacaoCentroCusto::where('conta_pagar_id', $id)
            ->with('centroCusto')
            ->get();
            
        $anexos = AnexoConta::where('conta_pagar_id', $id)
            ->get();
            
        $periodicidades = ContaPagar::periodicidades();
        $prioridades = ContaPagar::prioridades();
        $niveisAprovacao = ContaPagar::niveisAprovacao();
        
        $parcelasGeradas = [];
        if ($item->recorrente) {
            $parcelasGeradas = ContaPagar::where('parent_id', $id)
                ->orderBy('parcela_atual')
                ->get();
        }

        return view('conta-pagar.edit', compact(
            'item', 
            'fornecedores', 
            'categorias', 
            'centrosCusto', 
            'alocacoes', 
            'anexos', 
            'periodicidades', 
            'prioridades', 
            'niveisAprovacao', 
            'parcelasGeradas'
        ));
    }

    public function update(Request $request, $id)
    {
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);
        
        try {
            DB::beginTransaction();
            
            $file_name = $item->arquivo;
            if ($request->hasFile('file')) {
                $this->uploadUtil->unlinkImage($item, '/financeiro');
                $file_name = $this->uploadUtil->uploadFile($request->file, '/financeiro');
            }
            
            // Tratar dados básicos da conta
            $request->merge([
                'valor_integral' => __convert_value_bd($request->valor_integral),
                'valor_pago' => __convert_value_bd($request->valor_pago) ? __convert_value_bd($request->valor_pago) : 0,
                'arquivo' => $file_name,
                'recorrente' => $request->has('recorrente'),
            ]);
            
            // Se o valor mudou, recalcular nível de aprovação apenas se ainda não aprovado
            if ($item->valor_integral != $request->valor_integral && !$item->isAprovada()) {
                $temp = new ContaPagar();
                $temp->valor_integral = $request->valor_integral;
                $nivel_aprovacao = $temp->defineNivelAprovacao();
                $request->merge(['nivel_aprovacao' => $nivel_aprovacao]);
            }
            
            // Atualizar a conta
            $item->fill($request->all())->save();
            
            // Atualizar alocações de centro de custo
            if ($request->has('centro_custo_valores') && is_array($request->centro_custo_valores)) {
                // Remover alocações existentes
                AlocacaoCentroCusto::where('conta_pagar_id', $id)->delete();
                
                // Adicionar novas alocações
                foreach ($request->centro_custo_valores as $index => $valor) {
                    if (!empty($valor)) {
                        $valorAlocacao = __convert_value_bd($valor);
                        $porcentagem = ($valorAlocacao / $request->valor_integral) * 100;
                        
                        AlocacaoCentroCusto::create([
                            'conta_pagar_id' => $item->id,
                            'centro_custo_id' => $request->centro_custo_ids[$index],
                            'valor' => $valorAlocacao,
                            'porcentagem' => $porcentagem,
                            'observacao' => $request->centro_custo_observacoes[$index] ?? null
                        ]);
                    }
                }
            }
            
            // Processar múltiplos anexos
            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $anexo) {
                    $nomeArquivo = uniqid() . '.' . $anexo->getClientOriginalExtension();
                    $caminhoArquivo = $anexo->storeAs('financeiro/anexos', $nomeArquivo, 'public');
                    
                    AnexoConta::create([
                        'conta_pagar_id' => $item->id,
                        'nome_arquivo' => $anexo->getClientOriginalName(),
                        'caminho_arquivo' => $caminhoArquivo,
                        'tipo_arquivo' => $anexo->getClientMimeType(),
                        'tamanho_arquivo' => $anexo->getSize(),
                        'usuario_id' => Auth::id()
                    ]);
                }
            }
            
            // Gerar novas parcelas se recorrência foi ativada e não havia parcelas anteriores
            if ($request->recorrente && 
                $request->periodicidade && 
                $request->total_parcelas > 0 && 
                !$item->contasRecorrentes()->exists()) {
                
                $item->periodicidade = $request->periodicidade;
                $item->total_parcelas = $request->total_parcelas;
                $item->parcela_atual = 1;
                $item->save();
                
                // Gerar parcelas futuras
                for ($i = 2; $i <= $request->total_parcelas; $i++) {
                    $parcela = clone $item;
                    $parcela->id = null;
                    $parcela->status = 0;
                    $parcela->valor_pago = 0;
                    $parcela->data_pagamento = null;
                    $parcela->aprovado_por = null;
                    $parcela->data_aprovacao = null;
                    $parcela->parent_id = $item->id;
                    $parcela->parcela_atual = $i;
                    
                    // Garantir que a descrição seja mantida
                    $parcela->descricao = $item->descricao;
                    
                    // Calcular próxima data de vencimento baseado na periodicidade
                    $dataBase = $item->data_vencimento;
                    switch ($request->periodicidade) {
                        case ContaPagar::PERIODICIDADE_DIARIA:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' days'));
                            break;
                        case ContaPagar::PERIODICIDADE_SEMANAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' weeks'));
                            break;
                        case ContaPagar::PERIODICIDADE_QUINZENAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 15) . ' days'));
                            break;
                        case ContaPagar::PERIODICIDADE_MENSAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_BIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 2) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_TRIMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 3) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_SEMESTRAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . (($i-1) * 6) . ' months'));
                            break;
                        case ContaPagar::PERIODICIDADE_ANUAL:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' years'));
                            break;
                        default:
                            $dataVencimento = date('Y-m-d', strtotime($dataBase . ' + ' . ($i-1) . ' months'));
                    }
                    
                    $parcela->data_vencimento = $dataVencimento;
                    $parcela->save();
                    
                    $descricaoLog = "Vencimento (parcela $i): " . __data_pt($dataVencimento, 0) . " R$ " . __moeda($item->valor_integral);
                    __createLog(request()->empresa_id, 'Conta a Pagar', 'cadastrar', $descricaoLog);
                }
            }
            // Se não é mais recorrente, marcar esse atributo nas parcelas existentes
            elseif (!$request->recorrente && $item->contasRecorrentes()->exists()) {
                $item->contasRecorrentes()->update(['recorrente' => false]);
            }

            $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento) . " R$ " . __moeda($item->valor_integral);
            __createLog(request()->empresa_id, 'Conta a Pagar', 'editar', $descricaoLog);
            
            DB::commit();
            session()->flash("flash_success", "Conta a pagar atualizada!");
        } catch (\Exception $e) {
            DB::rollBack();
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->route('conta-pagar.index');
    }

    public function downloadFile($id){
        $item = ContaPagar::findOrFail($id);
        if (file_exists(public_path('uploads/financeiro/') . $item->arquivo)) {
            return response()->download(public_path('uploads/financeiro/') . $item->arquivo);
        } else {
            session()->flash("flash_error", "Arquivo não encontrado");
            return redirect()->back();
        }
    }

    private function __validate(Request $request)
    {
        $rules = [
            'fornecedor_id' => 'nullable',
            'valor_integral' => 'required',
            'data_vencimento' => 'required',
            'status' => 'required',
            'tipo_pagamento' => 'required'
        ];
        $messages = [
            'valor_integral.required' => 'Campo obrigatório',
            'data_vencimento.required' => 'Campo obrigatório',
            'status.required' => 'Campo obrigatório',
            'tipo_pagamento.required' => 'Campo obrigatório'
        ];
        $this->validate($request, $rules, $messages);
    }

    public function destroy($id)
    {
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);
        try {
            $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento, 0) . " R$ " . __moeda($item->valor_integral);
            $item->delete();
            __createLog(request()->empresa_id, 'Conta a Pagar', 'excluir', $descricaoLog);
            session()->flash("flash_success", "Conta removida!");
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->route('conta-pagar.index');
    }

    public function destroySelecet(Request $request)
    {
        $removidos = 0;
        for($i=0; $i<sizeof($request->item_delete); $i++){
            $item = ContaPagar::findOrFail($request->item_delete[$i]);
            try {
                $descricaoLog = "Vencimento: " . __data_pt($item->data_vencimento, 0) . " R$ " . __moeda($item->valor_integral);
                $item->delete();
                $removidos++;
                __createLog(request()->empresa_id, 'Conta a Pagar', 'excluir', $descricaoLog);
            } catch (\Exception $e) {
                __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
                session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
                return redirect()->back();
            }
        }

        session()->flash("flash_success", "Total de itens removidos: $removidos");
        return redirect()->back();
    }

    public function pagarSelecionados(Request $request)
    {
        // Log para debug
        \Illuminate\Support\Facades\Log::info('Iniciando pagamento em lote', [
            'request_all' => $request->all(),
            'ids' => $request->ids_selecionados ?? $request->item_recebe_paga ?? 'não informado',
            'tipo_pagamento' => $request->tipo_pagamento ?? 'não informado',
            'route' => $request->route()->getName()
        ]);
        
        $pagos = 0;
        $erros = 0;
        
        try {
            // Verificar se o caixa está aberto
            if (!__isCaixaAberto()) {
                session()->flash("flash_warning", "Abrir caixa antes de continuar!");
                return redirect()->route('caixa.create');
            }
            
            // Obter usuário e caixa
            $usuario = Auth::user()->id;
            $caixa = Caixa::where('usuario_id', $usuario)->where('status', 1)->first();
            
            if (!$caixa) {
                throw new \Exception("Caixa não encontrado para o usuário atual.");
            }
            
            // Verificar se ids foram passados - podem vir de duas formas diferentes
            $idsToProcess = [];
            
            if (!empty($request->ids_selecionados)) {
                // Do modal de pagamento em lote
                $idsToProcess = explode(',', $request->ids_selecionados);
            } elseif (!empty($request->item_recebe_paga)) {
                // Do formulário normal
                $idsToProcess = $request->item_recebe_paga;
            }
            
            if (empty($idsToProcess)) {
                throw new \Exception("Nenhuma conta selecionada para pagamento.");
            }
            
            \Illuminate\Support\Facades\Log::info('IDs para processar', [
                'qtd_ids' => count($idsToProcess),
                'ids' => $idsToProcess
            ]);
            
            // Se estiver usando valores personalizados
            $valores = [];
            if (!empty($request->valores_pagamento)) {
                $valores = explode(',', $request->valores_pagamento);
            }
            
            // Verificar o tipo de pagamento
            if (empty($request->tipo_pagamento)) {
                $tipo_pagamento = 'dinheiro'; // Default
            } else {
                $tipo_pagamento = $request->tipo_pagamento;
            }
            
            // Data do pagamento
            $data_pagamento = $request->data_pagamento ?? date('Y-m-d');
            
            // Processar cada conta
            foreach ($idsToProcess as $index => $id) {
                try {
                    // Buscar a conta com tratamento de erro
                    $item = ContaPagar::find($id);
                    
                    \Illuminate\Support\Facades\Log::info('Processando conta', [
                        'id' => $id,
                        'encontrada' => $item ? 'sim' : 'não'
                    ]);
                    
                    if (!$item) {
                        \Illuminate\Support\Facades\Log::warning("Conta ID {$id} não encontrada");
                        $erros++;
                        continue;
                    }
                    
                    // Validar se pertence à empresa atual
                    if ($item->empresa_id != request()->empresa_id) {
                        \Illuminate\Support\Facades\Log::warning("Conta ID {$id} não pertence à empresa atual");
                        $erros++;
                        continue;
                    }
                    
                    // Se a conta já estiver paga, pula
                    if ($item->status) {
                        \Illuminate\Support\Facades\Log::info("Conta ID {$id} já está paga");
                        continue;
                    }
                    
                    // Definir valor a ser pago
                    if (isset($valores[$index])) {
                        $valor_pago = (float) $valores[$index];
                    } else {
                        $valor_pago = (float) $item->valor_integral - (float) ($item->valor_pago ?? 0);
                    }
                    
                    // Registrar pagamento
                    $valor_anterior = (float) ($item->valor_pago ?? 0);
                    $item->valor_pago = $valor_anterior + $valor_pago;
                    $item->data_pagamento = $data_pagamento;
                    $item->tipo_pagamento = $tipo_pagamento;
                    $item->caixa_id = $caixa->id;
                    
                    // Verificar se é pagamento total
                    $valor_integral = (float) $item->valor_integral;
                    if ($item->valor_pago >= $valor_integral) {
                        $item->status = 1; // Pago
                    }
                    
                    $item->save();
                    
                    // Log de sucesso
                    $fornecedor = $item->fornecedor ? $item->fornecedor->razao_social : "ID {$item->fornecedor_id}";
                    \Illuminate\Support\Facades\Log::info("Conta ID {$id} paga com sucesso", [
                        'fornecedor' => $fornecedor,
                        'valor' => $valor_pago
                    ]);
                    
                    $pagos++;
                    
                    // Registrar log no sistema
                    $descricaoLog = "Pagamento de R$ " . __moeda($valor_pago) . " para " . $fornecedor;
                    __createLog(request()->empresa_id, 'Conta a Pagar', 'pagamento', $descricaoLog);
                    
                } catch (\Exception $e) {
                    \Illuminate\Support\Facades\Log::error("Erro ao processar pagamento da conta ID {$id}", [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                    $erros++;
                }
            }
            
            // Mensagem de resultado
            if ($pagos > 0) {
                session()->flash("flash_success", "Total de contas pagas: {$pagos}");
            }
            
            if ($erros > 0) {
                session()->flash("flash_warning", "Não foi possível processar {$erros} conta(s). Verifique os logs para mais detalhes.");
            }
            
            if ($pagos == 0 && $erros == 0) {
                session()->flash("flash_info", "Nenhuma conta foi processada.");
            }
            
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error("Erro geral no processamento de pagamentos", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            session()->flash("flash_error", "Erro ao processar pagamentos: " . $e->getMessage());
        }
        
        return redirect()->back();
    }

    public function pay($id)
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }
        $item = ContaPagar::findOrFail($id);

        if($item->status){
            session()->flash("flash_warning", "Esta conta já esta paga!");
            return redirect()->route('conta-pagar.index');
        }
        return view('conta-pagar.pay', compact('item'));
    }

    public function payPut(Request $request, $id)
    {
        $usuario = Auth::user()->id;
        $caixa = Caixa::where('usuario_id', $usuario)->where('status', 1)->first();
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);

        try {
            $valorPago = __convert_value_bd($request->valor_pago);
            $valorPendente = $item->valor_integral - ($item->valor_pago ?? 0);
            
            // Verificar se é pagamento parcial
            $valorMenor = $valorPago < $valorPendente;
            
            // Verificar se a conta está aprovada, se necessário
            if (!$item->isAprovada() && $item->nivel_aprovacao != null) {
                session()->flash("flash_warning", "Esta conta requer aprovação antes do pagamento!");
                return redirect()->route('conta-pagar.index');
            }
            
            // Se for pagamento total
            if (!$valorMenor) {
                $item->valor_pago = ($item->valor_pago ?? 0) + $valorPago;
                $item->status = true; // marca como completamente pago
                $item->data_pagamento = $request->data_pagamento;
                $item->tipo_pagamento = $request->tipo_pagamento;
                $item->caixa_id = $caixa->id;
                $item->save();
                
                $mensagemSucesso = "Conta paga integralmente!";
            } else {
                // Se for pagamento parcial
                // Registra o pagamento parcial
                $item->valor_pago = ($item->valor_pago ?? 0) + $valorPago;
                $item->status = false; // mantém como não totalmente pago
                $item->data_pagamento = null; // a data final de pagamento só é preenchida quando totalmente pago
                $item->tipo_pagamento = $request->tipo_pagamento;
                $item->caixa_id = $caixa->id;
                $item->save();
                
                $mensagemSucesso = "Pagamento parcial registrado! Valor restante: R$ " . 
                    __moeda($item->valor_integral - $item->valor_pago);
            }

            // Registrar saída no caixa/conta se especificado
            if(isset($request->conta_empresa_id)){
                $data = [
                    'conta_id' => $request->conta_empresa_id,
                    'descricao' => ($valorMenor ? "Pagamento parcial" : "Pagamento integral") . " da conta " . $item->id,
                    'tipo_pagamento' => $request->tipo_pagamento,
                    'valor' => $valorPago,
                    'tipo' => 'saida'
                ];
                $itemContaEmpresa = ItemContaEmpresa::create($data);
                $this->util->atualizaSaldo($itemContaEmpresa);
            }

            // Registra histórico da transação
            $descricaoLog = "Pagamento " . ($valorMenor ? "parcial" : "integral") . 
                " de R$ " . __moeda($valorPago) . 
                " para fornecedor: " . $item->fornecedor->razao_social;
            __createLog(request()->empresa_id, 'Conta a Pagar', 'pagamento', $descricaoLog);

            session()->flash("flash_success", $mensagemSucesso);
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Erro ao processar pagamento: " . $e->getMessage());
        }
        
        return redirect()->route('conta-pagar.index');
    }

    public function approve($id)
    {
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);
        
        if ($item->isAprovada()) {
            session()->flash("flash_warning", "Esta conta já está aprovada!");
            return redirect()->route('conta-pagar.index');
        }
        
        // Verificar se o usuário tem permissão para aprovar este nível
        $usuario = Auth::user();
        $nivel = $item->nivel_aprovacao;
        
        $podeAprovar = false;
        
        // Se for administrador, pode aprovar qualquer nível
        if ($usuario->admin == 1) {
            $podeAprovar = true;
        }
        // Verificar por nível específico usando funções auxiliares
        elseif ($nivel == ContaPagar::NIVEL_COORDENADOR && __canApproveAsCoordinator()) {
            $podeAprovar = true;
        }
        elseif ($nivel == ContaPagar::NIVEL_GERENTE && __canApproveAsManager()) {
            $podeAprovar = true;
        }
        elseif ($nivel == ContaPagar::NIVEL_DIRETOR && __canApproveAsDirector()) {
            $podeAprovar = true;
        }
        
        if (!$podeAprovar) {
            session()->flash("flash_error", "Você não tem permissão para aprovar contas deste nível ($nivel)!");
            return redirect()->route('conta-pagar.index');
        }
        
        try {
            $item->aprovado_por = $usuario->id;
            $item->data_aprovacao = now();
            $item->save();
            
            $descricaoLog = "Aprovação de conta a pagar: R$ " . __moeda($item->valor_integral) . " - " . $item->descricao;
            __createLog(request()->empresa_id, 'Conta a Pagar', 'aprovar', $descricaoLog);
            
            session()->flash("flash_success", "Conta a pagar aprovada com sucesso!");
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        
        return redirect()->route('conta-pagar.index');
    }
    
    public function removeApproval($id)
    {
        $item = ContaPagar::findOrFail($id);
        __validaObjetoEmpresa($item);
        
        if (!$item->isAprovada()) {
            session()->flash("flash_warning", "Esta conta não está aprovada!");
            return redirect()->route('conta-pagar.index');
        }
        
        // Verificar se o usuário tem permissão para remover aprovação
        if (!__canRemoveApproval()) {
            session()->flash("flash_error", "Você não tem permissão para remover aprovações!");
            return redirect()->route('conta-pagar.index');
        }
        
        try {
            $item->aprovado_por = null;
            $item->data_aprovacao = null;
            $item->save();
            
            $descricaoLog = "Remoção de aprovação de conta a pagar: R$ " . __moeda($item->valor_integral) . " - " . $item->descricao;
            __createLog(request()->empresa_id, 'Conta a Pagar', 'remover_aprovacao', $descricaoLog);
            
            session()->flash("flash_success", "Aprovação removida com sucesso!");
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Conta a Pagar', 'erro', $e->getMessage());
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        
        return redirect()->route('conta-pagar.index');
    }
}

