<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Consignacao;
use App\Models\ItemConsignacao;
use App\Models\Cliente;
use App\Models\Funcionario;
use App\Models\Produto;
use App\Models\CategoriaProduto;
use App\Models\Marca;
use App\Models\Caixa;
use App\Models\Empresa;
use App\Models\ConfigGeral;
use App\Models\Nfce;
use App\Models\ComprovanteNaoFiscal;
use App\Models\ItemNfce;
use App\Models\FaturaNfce;
use App\Utils\EstoqueUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class ConsignacaoController extends Controller
{
    protected $estoqueUtil;

    public function __construct(EstoqueUtil $estoqueUtil)
    {
        $this->estoqueUtil = $estoqueUtil;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $consignacoes = Consignacao::with(['cliente', 'funcionario', 'usuario'])
            ->where('empresa_id', request()->empresa_id)
            ->when($request->cliente_id, function($query, $cliente_id) {
                return $query->where('cliente_id', $cliente_id);
            })
            ->when($request->funcionario_id, function($query, $funcionario_id) {
                return $query->where('funcionario_id', $funcionario_id);
            })
            ->when($request->status, function($query, $status) {
                return $query->where('status', $status);
            })
            ->when($request->start_date, function($query, $start_date) {
                return $query->whereDate('data_consignacao', '>=', $start_date);
            })
            ->when($request->end_date, function($query, $end_date) {
                return $query->whereDate('data_consignacao', '<=', $end_date);
            })
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $clientes = Cliente::where('empresa_id', request()->empresa_id)->get();
        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();

        return view('consignacao.index', compact('consignacoes', 'clientes', 'funcionarios'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }

        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();
        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();
        $marcas = Marca::where('empresa_id', request()->empresa_id)->orderBy('nome', 'asc')->get();
        $caixa = __isCaixaAberto();
        $config = ConfigGeral::where('empresa_id', request()->empresa_id)->first();

        $view = 'consignacao.create';
        $produtos = [];

        if($config != null && $config->modelo == 'compact'){
            $view = 'consignacao.create2';
            $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)
            ->where('categoria_id', null)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $marcas = Marca::where('empresa_id', request()->empresa_id)
            ->orderBy('nome', 'asc')
            ->paginate(4);

            $produtos = Produto::select('produtos.*', \DB::raw('sum(quantidade) as quantidade'))
            ->where('empresa_id', request()->empresa_id)
            ->where('produtos.status', 1)
            ->where('status', 1)
            ->leftJoin('item_nfces', 'item_nfces.produto_id', '=', 'produtos.id')
            ->groupBy('produtos.id')
            ->orderBy('quantidade', 'desc')
            ->paginate(12);
        }

        $local_id = $caixa->local_id;

        return view($view, compact('categorias', 'funcionarios', 'caixa', 'config', 'marcas', 'produtos', 'local_id'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Decodifica itens_json se existir e popula o array 'itens' para a validação e processamento
        if ($request->filled('itens_json')) {
            $itens = json_decode($request->input('itens_json'), true);
            $request->merge(['itens' => $itens]);
        }

        try {
            $request->validate([
                'cliente_id' => 'required|exists:clientes,id',
                'funcionario_id' => 'required|exists:funcionarios,id',
                'data_vencimento' => 'nullable|date|after:today',
                'observacao' => 'nullable|string|max:500',
                'itens' => 'required|array|min:1',
                'itens.*.produto_id' => 'required|exists:produtos,id',
                'itens.*.quantidade' => 'required|numeric|min:0.001',
                'itens.*.valor_unitario' => 'required|numeric|min:0.01',
            ], [
                'data_vencimento.after' => 'A data de vencimento deve ser maior que hoje.',
                'cliente_id.required' => 'O cliente é obrigatório.',
                'cliente_id.exists' => 'Cliente selecionado não existe.',
                'funcionario_id.required' => 'O funcionário é obrigatório.',
                'funcionario_id.exists' => 'Funcionário selecionado não existe.',
                'itens.required' => 'É necessário adicionar pelo menos um item.',
                'itens.min' => 'É necessário adicionar pelo menos um item.',
                'itens.*.produto_id.required' => 'Produto é obrigatório.',
                'itens.*.produto_id.exists' => 'Produto selecionado não existe.',
                'itens.*.quantidade.required' => 'Quantidade é obrigatória.',
                'itens.*.quantidade.numeric' => 'Quantidade deve ser um número.',
                'itens.*.quantidade.min' => 'Quantidade deve ser maior que zero.',
                'itens.*.valor_unitario.required' => 'Valor unitário é obrigatório.',
                'itens.*.valor_unitario.numeric' => 'Valor unitário deve ser um número.',
                'itens.*.valor_unitario.min' => 'Valor unitário deve ser maior que zero.',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Preservar os dados dos itens para restaurar no formulário
            if ($request->has('itens_json')) {
                session()->flash('itens_json', $request->itens_json);
            }
            throw $e;
        }

        try {
            DB::beginTransaction();

            // Gerar número da consignação
            $numero = $this->gerarNumeroConsignacao();

            $consignacao = Consignacao::create([
                'empresa_id' => request()->empresa_id,
                'usuario_id' => Auth::user()->id,
                'cliente_id' => $request->cliente_id,
                'funcionario_id' => $request->funcionario_id,
                'numero_consignacao' => $numero,
                'data_consignacao' => now(),
                'data_vencimento' => $request->data_vencimento,
                'valor_total' => 0,
                'observacao' => $request->observacao,
                'status' => 'ativa',
                'local_id' => __isCaixaAberto()->local_id
            ]);

            $valorTotal = 0;

            foreach ($request->itens as $item) {
                $subTotal = $item['quantidade'] * $item['valor_unitario'];
                $valorTotal += $subTotal;

                ItemConsignacao::create([
                    'consignacao_id' => $consignacao->id,
                    'produto_id' => $item['produto_id'],
                    'produto_variacao_id' => $item['produto_variacao_id'] ?? null,
                    'quantidade' => $item['quantidade'],
                    'quantidade_vendida' => 0,
                    'quantidade_devolvida' => 0,
                    'valor_unitario' => $item['valor_unitario'],
                    'sub_total' => $subTotal,
                    'observacao' => $item['observacao'] ?? null
                ]);

                // Baixar estoque
                $this->estoqueUtil->reduzEstoque(
                    $item['produto_id'],
                    $item['quantidade'],
                    $item['produto_variacao_id'] ?? null,
                    $consignacao->local_id ?? null
                );
                // Registrar movimentação de estoque
                $this->estoqueUtil->movimentacaoProduto(
                    $item['produto_id'],
                    $item['quantidade'],
                    'saida',
                    $consignacao->id,
                    'consignacao_saida',
                    Auth::id(),
                    $item['produto_variacao_id'] ?? null
                );
            }

            $consignacao->update(['valor_total' => $valorTotal]);

            DB::commit();

            session()->flash('flash_success', 'Consignação criada com sucesso!');
            return redirect()->route('consignacao.show', $consignacao->id);

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('flash_error', 'Erro ao criar consignação: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $consignacao = Consignacao::with(['cliente', 'funcionario', 'usuario', 'itens.produto', 'itens.produtoVariacao'])
            ->where('empresa_id', request()->empresa_id)
            ->findOrFail($id);

        return view('consignacao.show', compact('consignacao'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $consignacao = Consignacao::with(['itens.produto', 'itens.produtoVariacao'])
            ->where('empresa_id', request()->empresa_id)
            ->findOrFail($id);

        if (!$consignacao->podeSerEditada()) {
            session()->flash('flash_error', 'Esta consignação não pode ser editada.');
            return redirect()->route('consignacao.show', $consignacao->id);
        }

        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();
        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();
        $caixa = __isCaixaAberto();
        $config = ConfigGeral::where('empresa_id', request()->empresa_id)->first();

        return view('consignacao.edit', compact('consignacao', 'categorias', 'funcionarios', 'caixa', 'config'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $consignacao = Consignacao::where('empresa_id', request()->empresa_id)->findOrFail($id);

        if (!$consignacao->podeSerEditada()) {
            session()->flash('flash_error', 'Esta consignação não pode ser editada.');
            return redirect()->route('consignacao.show', $consignacao->id);
        }

        // Se for apenas atualização de quantidade devolvida
        if ($request->has('itens') && $request->has('itens.0.quantidade_devolvida')) {
            return $this->atualizarQuantidadeDevolvida($request, $consignacao);
        }

        $request->validate([
            'cliente_id' => 'required|exists:clientes,id',
            'funcionario_id' => 'required|exists:funcionarios,id',
            'data_vencimento' => 'nullable|date',
            'observacao' => 'nullable|string|max:500',
            'itens' => 'required|array|min:1',
            'itens.*.produto_id' => 'required|exists:produtos,id',
            'itens.*.quantidade' => 'required|numeric|min:0.001',
            'itens.*.valor_unitario' => 'required|numeric|min:0.01',
        ], [
            'cliente_id.required' => 'O cliente é obrigatório.',
            'cliente_id.exists' => 'Cliente selecionado não existe.',
            'funcionario_id.required' => 'O funcionário é obrigatório.',
            'funcionario_id.exists' => 'Funcionário selecionado não existe.',
            'itens.required' => 'É necessário adicionar pelo menos um item.',
            'itens.min' => 'É necessário adicionar pelo menos um item.',
            'itens.*.produto_id.required' => 'Produto é obrigatório.',
            'itens.*.produto_id.exists' => 'Produto selecionado não existe.',
            'itens.*.quantidade.required' => 'Quantidade é obrigatória.',
            'itens.*.quantidade.numeric' => 'Quantidade deve ser um número.',
            'itens.*.quantidade.min' => 'Quantidade deve ser maior que zero.',
            'itens.*.valor_unitario.required' => 'Valor unitário é obrigatório.',
            'itens.*.valor_unitario.numeric' => 'Valor unitário deve ser um número.',
            'itens.*.valor_unitario.min' => 'Valor unitário deve ser maior que zero.',
        ]);

        try {
            DB::beginTransaction();

            // Restaurar estoque dos itens antigos
            foreach ($consignacao->itens as $item) {
                $this->estoqueUtil->adicionarEstoque(
                    $item->produto_id,
                    $item->produto_variacao_id,
                    $item->quantidade,
                    'consignacao',
                    $consignacao->id,
                    'consignacao_edicao_restauracao'
                );
            }

            // Atualizar dados da consignação
            $consignacao->update([
                'cliente_id' => $request->cliente_id,
                'funcionario_id' => $request->funcionario_id,
                'data_vencimento' => $request->data_vencimento,
                'observacao' => $request->observacao,
            ]);

            // Remover itens antigos
            $consignacao->itens()->delete();

            $valorTotal = 0;

            // Adicionar novos itens
            foreach ($request->itens as $item) {
                $subTotal = $item['quantidade'] * $item['valor_unitario'];
                $valorTotal += $subTotal;

                ItemConsignacao::create([
                    'consignacao_id' => $consignacao->id,
                    'produto_id' => $item['produto_id'],
                    'produto_variacao_id' => $item['produto_variacao_id'] ?? null,
                    'quantidade' => $item['quantidade'],
                    'quantidade_vendida' => 0,
                    'quantidade_devolvida' => 0,
                    'valor_unitario' => $item['valor_unitario'],
                    'sub_total' => $subTotal,
                    'observacao' => $item['observacao'] ?? null
                ]);

                // Baixar estoque
                $this->estoqueUtil->baixarEstoque(
                    $item['produto_id'],
                    $item['produto_variacao_id'] ?? null,
                    $item['quantidade'],
                    'consignacao',
                    $consignacao->id,
                    'consignacao_edicao_saida'
                );
            }

            $consignacao->update(['valor_total' => $valorTotal]);

            DB::commit();

            session()->flash('flash_success', 'Consignação atualizada com sucesso!');
            return redirect()->route('consignacao.show', $consignacao->id);

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('flash_error', 'Erro ao atualizar consignação: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Atualizar apenas quantidade devolvida dos itens
     */
    private function atualizarQuantidadeDevolvida(Request $request, $consignacao)
    {
        $request->validate([
            'itens' => 'required|array|min:1',
            'itens.*.item_id' => 'required|exists:item_consignacoes,id',
            'itens.*.quantidade_devolvida' => 'required|numeric|min:0',
            'observacao' => 'nullable|string|max:500',
        ]);

        try {
            DB::beginTransaction();

            foreach ($request->itens as $item) {
                $itemConsignacao = $consignacao->itens()->findOrFail($item['item_id']);
                
                // Validar se a quantidade devolvida não excede a vendida
                if ($item['quantidade_devolvida'] > $itemConsignacao->quantidade_vendida) {
                    throw new \Exception('Quantidade devolvida não pode ser maior que a quantidade vendida.');
                }

                $itemConsignacao->update([
                    'quantidade_devolvida' => $item['quantidade_devolvida']
                ]);
            }

            // Atualizar observação se fornecida
            if ($request->observacao) {
                $consignacao->update(['observacao' => $request->observacao]);
            }

            DB::commit();

            session()->flash('flash_success', 'Quantidades devolvidas atualizadas com sucesso!');
            return redirect()->route('consignacao.show', $consignacao->id);

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('flash_error', 'Erro ao atualizar quantidades devolvidas: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $consignacao = Consignacao::where('empresa_id', request()->empresa_id)->findOrFail($id);

        if (!$consignacao->podeSerCancelada()) {
            session()->flash('flash_error', 'Esta consignação não pode ser cancelada.');
            return redirect()->back();
        }

        try {
            DB::beginTransaction();

            // Restaurar estoque
            foreach ($consignacao->itens as $item) {
                $this->estoqueUtil->incrementaEstoque(
                    $item->produto_id,
                    $item->quantidade,
                    $item->produto_variacao_id
                );
            }

            $consignacao->update(['status' => 'cancelada']);

            DB::commit();

            session()->flash('flash_success', 'Consignação cancelada com sucesso!');
            return redirect()->route('consignacao.index');

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('flash_error', 'Erro ao cancelar consignação: ' . $e->getMessage());
            return redirect()->back();
        }
    }

    /**
     * Converter consignação em venda (NFCe ou Comprovante Não Fiscal)
     */
    public function converterEmVenda(Request $request, string $id)
    {
        $consignacao = Consignacao::with(['itens.produto', 'itens.produtoVariacao', 'cliente', 'funcionario'])
            ->where('empresa_id', request()->empresa_id)
            ->findOrFail($id);

        if (!$consignacao->podeSerFinalizada()) {
            session()->flash('flash_error', 'Esta consignação não pode ser convertida em venda.');
            return redirect()->back();
        }

        $request->validate([
            'tipo_documento' => 'required|in:nfce,comprovante',
            'forma_pagamento' => 'required|string',
            'condicao_pagamento' => 'required|in:avista,prazo',
            'parcelas' => 'required|integer|min:1|max:12',
            'observacao_venda' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            if ($request->tipo_documento === 'nfce') {
                $this->converterEmNfce($consignacao, $request);
            } else {
                $this->converterEmComprovante($consignacao, $request);
            }

            $consignacao->update(['status' => 'finalizada']);

            DB::commit();

            session()->flash('flash_success', 'Consignação convertida em venda com sucesso!');
            return redirect()->route('consignacao.show', $consignacao->id);

        } catch (\Exception $e) {
            DB::rollback();
            session()->flash('flash_error', 'Erro ao converter consignação: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Registrar devolução de itens
     */
    public function registrarDevolucao(Request $request, $id)
    {
        $consignacao = Consignacao::with(['itens'])->where('empresa_id', request()->empresa_id)->findOrFail($id);
        DB::beginTransaction();
        try {
            foreach ($request->input('itens', []) as $itemId => $itemData) {
                $item = $consignacao->itens->where('id', $itemId)->first();
                if (!$item) continue;
                $novaDevolucao = floatval($itemData['quantidade'] ?? 0);
                if ($novaDevolucao > 0 && $novaDevolucao <= ($item->quantidade - $item->quantidade_devolvida)) {
                    $item->quantidade_devolvida += $novaDevolucao;
                    $item->save();
                    // Atualizar estoque
                    $this->estoqueUtil->incrementaEstoque($item->produto_id, $novaDevolucao, $item->produto_variacao_id);
                    // Registrar movimentação de estoque
                    $this->estoqueUtil->movimentacaoProduto(
                        $item->produto_id,
                        $novaDevolucao,
                        'entrada',
                        $consignacao->id,
                        'devolucao_consignacao',
                        Auth::id(),
                        $item->produto_variacao_id
                    );
                }
            }
            // Se todos os itens foram devolvidos, marcar como finalizada
            $todosDevolvidos = $consignacao->itens->every(function($item) {
                return $item->quantidade_devolvida >= $item->quantidade;
            });
            if ($todosDevolvidos) {
                $consignacao->status = 'finalizada';
                $consignacao->save();
            }
            DB::commit();
            session()->flash('flash_success', 'Devolução registrada com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('flash_error', 'Erro ao registrar devolução: ' . $e->getMessage());
        }
        return redirect()->route('frontbox.create', ['consignacao_id' => $consignacao->id]);
    }

    /**
     * Imprimir comprovante de consignação
     */
    public function imprimir(string $id)
    {
        $consignacao = Consignacao::with(['cliente', 'funcionario', 'usuario', 'itens.produto', 'itens.produtoVariacao'])
            ->where('empresa_id', request()->empresa_id)
            ->findOrFail($id);

        return view('consignacao.print', compact('consignacao'));
    }

    /**
     * Gerar número único para consignação
     */
    private function gerarNumeroConsignacao()
    {
        $ultima = \App\Models\Consignacao::where('empresa_id', request()->empresa_id)
            ->orderByDesc('id')
            ->first();
        if ($ultima && preg_match('/VC(\\d+)/', $ultima->numero_consignacao, $matches)) {
            $novoNumero = intval($matches[1]) + 1;
        } else {
            $novoNumero = 1;
        }
        return 'VC' . str_pad($novoNumero, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Converter consignação em NFCe
     */
    private function converterEmNfce($consignacao, $request)
    {
        // Verificar se há itens vendidos
        $itensVendidos = $consignacao->itens->where('quantidade_vendida', '>', 0);
        
        if ($itensVendidos->isEmpty()) {
            throw new \Exception('Não há itens vendidos para converter em NFCe.');
        }

        // Criar NFCe
        $nfce = Nfce::create([
            'empresa_id' => request()->empresa_id,
            'usuario_id' => Auth::user()->id,
            'cliente_id' => $consignacao->cliente_id,
            'numero_nfce' => $this->gerarNumeroNfce(),
            'data_emissao' => now(),
            'valor_total' => $itensVendidos->sum(function($item) {
                return $item->quantidade_vendida * $item->valor_unitario;
            }),
            'forma_pagamento' => $request->forma_pagamento,
            'condicao_pagamento' => $request->condicao_pagamento,
            'parcelas' => $request->parcelas,
            'observacao' => $request->observacao_venda,
            'status' => 'pendente',
            'local_id' => $consignacao->local_id,
            'consignacao_id' => $consignacao->id
        ]);

        // Criar itens da NFCe
        foreach ($itensVendidos as $item) {
            ItemNfce::create([
                'nfce_id' => $nfce->id,
                'produto_id' => $item->produto_id,
                'produto_variacao_id' => $item->produto_variacao_id,
                'quantidade' => $item->quantidade_vendida,
                'valor_unitario' => $item->valor_unitario,
                'sub_total' => $item->quantidade_vendida * $item->valor_unitario,
                'descricao' => $item->descricao
            ]);
        }

        // Criar fatura da NFCe
        FaturaNfce::create([
            'nfce_id' => $nfce->id,
            'forma_pagamento' => $request->forma_pagamento,
            'valor' => $nfce->valor_total,
            'parcelas' => $request->parcelas
        ]);
    }

    /**
     * Converter consignação em Comprovante Não Fiscal
     */
    private function converterEmComprovante($consignacao, $request)
    {
        // Verificar se há itens vendidos
        $itensVendidos = $consignacao->itens->where('quantidade_vendida', '>', 0);
        
        if ($itensVendidos->isEmpty()) {
            throw new \Exception('Não há itens vendidos para converter em comprovante.');
        }

        // Criar Comprovante Não Fiscal
        $comprovante = ComprovanteNaoFiscal::create([
            'empresa_id' => request()->empresa_id,
            'usuario_id' => Auth::user()->id,
            'cliente_id' => $consignacao->cliente_id,
            'numero_comprovante' => $this->gerarNumeroComprovante(),
            'data_emissao' => now(),
            'valor_total' => $itensVendidos->sum(function($item) {
                return $item->quantidade_vendida * $item->valor_unitario;
            }),
            'forma_pagamento' => $request->forma_pagamento,
            'condicao_pagamento' => $request->condicao_pagamento,
            'parcelas' => $request->parcelas,
            'observacao' => $request->observacao_venda,
            'status' => 'finalizado',
            'local_id' => $consignacao->local_id,
            'consignacao_id' => $consignacao->id
        ]);

        // Aqui você pode criar os itens do comprovante se houver uma tabela específica
        // Por enquanto, vamos apenas registrar o comprovante
    }

    /**
     * Gerar número único para NFCe
     */
    private function gerarNumeroNfce()
    {
        $ano = date('Y');
        $mes = date('m');
        
        $ultimaNfce = Nfce::where('empresa_id', request()->empresa_id)
            ->whereYear('created_at', $ano)
            ->whereMonth('created_at', $mes)
            ->orderBy('id', 'desc')
            ->first();

        $sequencial = $ultimaNfce ? intval(substr($ultimaNfce->numero_nfce, -6)) + 1 : 1;
        
        return sprintf('NFCe%s%s%06d', $ano, $mes, $sequencial);
    }

    /**
     * Gerar número único para Comprovante
     */
    private function gerarNumeroComprovante()
    {
        $ano = date('Y');
        $mes = date('m');
        
        $ultimoComprovante = ComprovanteNaoFiscal::where('empresa_id', request()->empresa_id)
            ->whereYear('created_at', $ano)
            ->whereMonth('created_at', $mes)
            ->orderBy('id', 'desc')
            ->first();

        $sequencial = $ultimoComprovante ? intval(substr($ultimoComprovante->numero_comprovante, -6)) + 1 : 1;
        
        return sprintf('COMP%s%s%06d', $ano, $mes, $sequencial);
    }

    public function devolucao($id)
    {
        $consignacao = \App\Models\Consignacao::with(['itens.produto', 'cliente', 'funcionario'])
            ->where('empresa_id', request()->empresa_id)
            ->findOrFail($id);
        return view('consignacao.devolucao', compact('consignacao'));
    }
}
