<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Nfce;
use App\Models\Troca;
use App\Models\Produto;
use App\Models\ItemTroca;
use App\Models\Cliente;
use App\Models\CreditoCliente;
use Illuminate\Support\Str;
use App\Utils\EstoqueUtil;

class TrocaController extends Controller
{

    protected $util;

    public function __construct(EstoqueUtil $util)
    {
        $this->util = $util;
    }

    private function getLastNumero($empresa_id){
        $last = Troca::where('empresa_id', $empresa_id)
        ->orderBy('numero_sequencial', 'desc')
        ->where('numero_sequencial', '>', 0)->first();
        $numero = $last != null ? $last->numero_sequencial : 0;
        $numero++;
        return $numero;
    }

    public function store(Request $request){
        $nfce = Nfce::findOrFail($request->venda_id);
        try{

            $troca = Troca::create([
                'empresa_id' => $request->empresa_id,
                'nfce_id' => $request->venda_id,
                'observacao' => '',
                'numero_sequencial' => $this->getLastNumero($request->empresa_id),
                'codigo' => Str::random(8),
                'valor_troca' => __convert_value_bd($request->valor_total),
                'valor_original' => $nfce->total,
                'tipo_pagamento' => $request->tipo_pagamento ? $request->tipo_pagamento : $nfce->tipo_pagamento
            ]);

            // Log para debug da troca criada
            \Log::info("Troca criada:", [
                'id' => $troca->id,
                'tipo_pagamento' => $troca->tipo_pagamento,
                'valor_troca' => $troca->valor_troca,
                'valor_original' => $troca->valor_original
            ]);

            $nfce->total = __convert_value_bd($request->valor_total);
            $nfce->save();

            foreach($nfce->itens as $i){
                if ($i->produto->gerenciar_estoque) {
                    $this->util->incrementaEstoque($i->produto_id, $i->quantidade, null, $nfce->local_id);
                    
                    // Registrar movimentação de estoque (histórico)
                    $this->util->movimentacaoProduto(
                        $i->produto_id, 
                        $i->quantidade, 
                        'incremento', 
                        $nfce->id, 
                        'troca_devolucao', 
                        null,
                        null
                    );
                }
            }

            if($request->produto_id){
                for ($i = 0; $i < sizeof($request->produto_id); $i++) {
                    $produto_id = $request->produto_id[$i];
                    $quantidade = __convert_value_bd($request->quantidade[$i]);
                    $add = 1;
                    $qtd = 0;
                    foreach($nfce->itens as $itemNfce){
                        if($itemNfce->produto_id == $produto_id && $itemNfce->quantidade == $quantidade){
                            $add = 0;
                        }else{
                            if($itemNfce->produto_id == $produto_id && $itemNfce->quantidade != $quantidade){
                                $quantidade -= $itemNfce->quantidade;
                            }
                        }
                    }

                    if($add == 1){
                        ItemTroca::create([
                            'produto_id' => $produto_id,
                            'quantidade' => $quantidade,
                            'troca_id' => $troca->id
                        ]);
                    }
                    $product = Produto::findOrFail($produto_id);
                    if ($product->gerenciar_estoque) {
                        $this->util->reduzEstoque($product->id, $quantidade, null, $troca->nfce->local_id);
                        
                        // Registrar movimentação de estoque (histórico)
                        $this->util->movimentacaoProduto(
                            $product->id, 
                            $quantidade, 
                            'reducao', 
                            $troca->id, 
                            'troca_novo_produto', 
                            null,
                            null
                        );
                    }
                }
            }

            // Log para debug dos dados recebidos
            \Log::info("Dados da troca recebidos:", [
                'valor_credito' => $request->valor_credito,
                'cliente_id' => $request->cliente_id,
                'tipo_pagamento' => $request->tipo_pagamento,
                'tipo_pagamento_troca' => $troca->tipo_pagamento,
                'valor_total' => $request->valor_total,
                'valor_original_nfce' => $nfce->total
            ]);
            
            // Validar se deve processar crédito
            $valorCreditoOriginal = $request->valor_credito;
            $valorCreditoConvertido = __convert_value_bd($request->valor_credito);
            
            \Log::info("Validação de crédito:", [
                'valor_original' => $valorCreditoOriginal,
                'valor_convertido' => $valorCreditoConvertido,
                'valor_convertido_tipo' => gettype($valorCreditoConvertido),
                'cliente_id_existe' => !empty($request->cliente_id),
                'tipo_pagamento_00' => ($troca->tipo_pagamento == '00' || $request->tipo_pagamento == '00'),
                'condicao_1' => $valorCreditoConvertido > 0,
                'condicao_2' => !empty($request->cliente_id),
                'condicao_3' => ($troca->tipo_pagamento == '00' || $request->tipo_pagamento == '00')
            ]);
            
            // Verificar se todas as condições são atendidas
            $tipoPagamentoValido = ($troca->tipo_pagamento == '00' || $request->tipo_pagamento == '00');
            $valorCreditoValido = (float)$valorCreditoConvertido > 0;
            $clienteExiste = !empty($request->cliente_id);
            
            \Log::info("Validação detalhada do tipo de pagamento:", [
                'troca_tipo_pagamento' => $troca->tipo_pagamento,
                'request_tipo_pagamento' => $request->tipo_pagamento,
                'tipo_pagamento_valido' => $tipoPagamentoValido
            ]);
            
            \Log::info("Validação final das condições:", [
                'valor_credito_valido' => $valorCreditoValido,
                'cliente_existe' => $clienteExiste,
                'tipo_pagamento_valido' => $tipoPagamentoValido,
                'todas_condicoes' => ($valorCreditoValido && $clienteExiste && $tipoPagamentoValido)
            ]);
            
            if($valorCreditoValido && $clienteExiste && $tipoPagamentoValido){
                \Log::info("Todas as condições atendidas, processando crédito...");
                
                $cliente = Cliente::findOrFail($request->cliente_id);
                
                // Log para debug
                \Log::info("Processando crédito para cliente: " . $cliente->razao_social . " - Valor original: " . $valorCreditoOriginal . " - Valor convertido: " . $valorCreditoConvertido);
                
                try {
                    CreditoCliente::create([
                        'valor' => $valorCreditoConvertido,
                        'cliente_id' => $cliente->id,
                        'empresa_id' => $request->empresa_id
                    ]);

                    $cliente->valor_credito += $valorCreditoConvertido;
                    $cliente->save();
                    
                    // Log para confirmar que o crédito foi adicionado
                    \Log::info("Crédito adicionado com sucesso. Novo saldo: " . $cliente->valor_credito);
                } catch (\Exception $e) {
                    \Log::error("Erro ao processar crédito: " . $e->getMessage());
                    throw $e;
                }
            } else {
                \Log::info("Condições não atendidas para processar crédito:");
                \Log::info("- valorCreditoConvertido > 0: " . ($valorCreditoConvertido > 0 ? 'true' : 'false'));
                \Log::info("- cliente_id existe: " . (!empty($request->cliente_id) ? 'true' : 'false'));
                \Log::info("- tipo_pagamento é '00': " . (($troca->tipo_pagamento == '00' || $request->tipo_pagamento == '00') ? 'true' : 'false'));
            }
            __createLog($request->empresa_id, 'PDV Troca', 'cadastrar', "#$troca->numero_sequencial - R$ " . __moeda($troca->valor_troca));

            return response()->json($troca, 200);
        } catch (\Exception $e) {
            __createLog($request->empresa_id, 'PDV Troca', 'erro', $e->getMessage());
            return response()->json($e->getMessage() . ", line: " . $e->getLine() . ", file: " . $e->getFile(), 401);
        }
    }
}
