<?php

namespace App\Utils;

use NFePHP\Common\Certificate;
use NFePHP\Common\Exception\CertificateException;

class CertificateUtil
{
    /**
     * Detecta e carrega o certificado (A1 ou A3)
     * 
     * @param string $password Senha do certificado
     * @param string $type Tipo do certificado ('auto', 'a1', 'a3')
     * @return Certificate
     * @throws CertificateException
     */
    public static function loadCertificate($certificatePathOrContent, $password, $type = 'auto')
    {
        $content = null;
        // 1) Se for um caminho válido em disco, lê o arquivo normalmente
        if (is_string($certificatePathOrContent) && @file_exists($certificatePathOrContent)) {
            $content = @file_get_contents($certificatePathOrContent);
            if ($content === false || $content === '') {
                throw new CertificateException("Falha ao ler certificado em: {$certificatePathOrContent}");
            }
        } else {
            // 2) Caso contrário, trata a entrada como CONTEÚDO do certificado
            $raw = (string)$certificatePathOrContent;
            // Tenta resolver caminhos relativos comuns
            $looksLikePath = (strpos($raw, '/') !== false || strpos($raw, '\\') !== false || preg_match('/\.(pfx|p12|pem|cer|crt)$/i', $raw));
            if ($looksLikePath) {
                $candidates = [];
                $candidates[] = base_path($raw);
                $candidates[] = public_path($raw);
                $candidates[] = storage_path('app/'.ltrim($raw, '/\\'));
                foreach ($candidates as $cand) {
                    if (@file_exists($cand)) {
                        $fileContent = @file_get_contents($cand);
                        if ($fileContent !== false && $fileContent !== '') {
                            $content = $fileContent;
                            break;
                        }
                    }
                }
            }
            if ($content === null) {
            if (strpos($raw, '-----BEGIN') !== false) {
                // PEM puro
                $content = $raw;
            } else {
                // Tenta base64
                $maybeBase64 = preg_replace('/\s+/', '', $raw);
                $isBase64 = (bool) preg_match('/^[A-Za-z0-9+\/]+=*$/', $maybeBase64);
                if ($isBase64) {
                    $decoded = base64_decode($maybeBase64, true);
                    if ($decoded !== false && strlen($decoded) > 0) {
                        $content = $decoded;
                    }
                }
                if ($content === null) {
                    // Assume binário PFX/P12 já em memória
                    $content = $raw;
                }
            }
            }
        }

        if ($content === null || $content === '') {
            throw new CertificateException('Conteúdo do certificado vazio ou inválido');
        }

        // Se o tipo for 'auto', detecta automaticamente
        if ($type === 'auto') {
            $type = self::detectCertificateType($content, is_string($certificatePathOrContent) ? $certificatePathOrContent : '');
        }

        switch (strtolower($type)) {
            case 'a1':
                return self::loadA1Certificate($content, $password);
            case 'a3':
                return self::loadA3Certificate($content, $password);
            default:
                throw new CertificateException("Tipo de certificado não suportado: {$type}");
        }
    }

    /**
     * Detecta automaticamente o tipo de certificado
     * 
     * @param string $content Conteúdo do certificado
     * @param string $path Caminho do arquivo
     * @return string 'a1' ou 'a3'
     */
    private static function detectCertificateType($content, $path)
    {
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        
        // Por extensão do arquivo
        if (in_array($extension, ['pfx', 'p12'])) {
            return 'a1';
        }
        
        if (in_array($extension, ['pem', 'crt', 'cer'])) {
            return 'a3';
        }

        // Por conteúdo do arquivo
        if (strpos($content, '-----BEGIN CERTIFICATE-----') !== false) {
            return 'a3';
        }

        // Se o conteúdo for binário grande, assumimos A1 (PFX/P12)
        // Heurística simples: presença de muitos bytes não imprimíveis
        $nonPrintable = 0;
        $len = strlen($content);
        $check = substr($content, 0, min(512, $len));
        for ($i = 0; $i < strlen($check); $i++) {
            $ord = ord($check[$i]);
            if ($ord < 9 || ($ord > 13 && $ord < 32) || $ord > 126) {
                $nonPrintable++;
            }
        }
        if ($nonPrintable > 50) {
            return 'a1';
        }

        // Se não conseguir detectar, assume A1 (comportamento padrão)
        return 'a1';
    }

    /**
     * Carrega certificado A1 (PFX/P12)
     * 
     * @param string $content Conteúdo do certificado
     * @param string $password Senha
     * @return Certificate
     * @throws CertificateException
     */
    private static function loadA1Certificate($content, $password)
    {
        try {
            return Certificate::readPfx($content, $password);
        } catch (CertificateException $e) {
            throw new CertificateException("Erro ao carregar certificado A1: " . $e->getMessage());
        }
    }

    /**
     * Carrega certificado A3 (PEM)
     * 
     * @param string $content Conteúdo do certificado
     * @param string $password Senha
     * @return Certificate
     * @throws CertificateException
     */
    private static function loadA3Certificate($content, $password)
    {
        try {
            // Para certificados A3, precisamos extrair a chave privada e o certificado
            $certData = self::parsePemCertificate($content);
            
            if (!$certData) {
                throw new CertificateException("Formato de certificado A3 inválido");
            }

            // Criar certificado a partir dos componentes PEM
            return new Certificate(
                new \NFePHP\Common\Certificate\PrivateKey($certData['privateKey']),
                new \NFePHP\Common\Certificate\PublicKey($certData['certificate']),
                new \NFePHP\Common\Certificate\CertificationChain($certData['chain'] ?? '')
            );
        } catch (\Exception $e) {
            throw new CertificateException("Erro ao carregar certificado A3: " . $e->getMessage());
        }
    }

    /**
     * Faz o parse de um certificado PEM
     * 
     * @param string $content Conteúdo do certificado
     * @return array|false Array com os componentes ou false se inválido
     */
    private static function parsePemCertificate($content)
    {
        $sections = [];
        $currentSection = '';
        $currentData = '';

        $lines = explode("\n", $content);
        
        foreach ($lines as $line) {
            $line = trim($line);
            
            if (empty($line)) {
                continue;
            }

            // Verifica se é um cabeçalho de seção
            if (preg_match('/^-----BEGIN (.+)-----$/', $line, $matches)) {
                if (!empty($currentSection)) {
                    $sections[$currentSection] = trim($currentData);
                }
                $currentSection = strtolower($matches[1]);
                $currentData = '';
                continue;
            }

            // Verifica se é um rodapé de seção
            if (preg_match('/^-----END (.+)-----$/', $line)) {
                $sections[$currentSection] = trim($currentData);
                $currentSection = '';
                $currentData = '';
                continue;
            }

            // Adiciona linha aos dados da seção atual
            if (!empty($currentSection)) {
                $currentData .= $line . "\n";
            }
        }

        // Adiciona a última seção
        if (!empty($currentSection)) {
            $sections[$currentSection] = trim($currentData);
        }

        // Extrai os componentes necessários
        $result = [];
        
        // Certificado público
        if (isset($sections['certificate'])) {
            $result['certificate'] = "-----BEGIN CERTIFICATE-----\n" . $sections['certificate'] . "\n-----END CERTIFICATE-----";
        } elseif (isset($sections['x509 certificate'])) {
            $result['certificate'] = "-----BEGIN CERTIFICATE-----\n" . $sections['x509 certificate'] . "\n-----END CERTIFICATE-----";
        }

        // Chave privada
        if (isset($sections['private key'])) {
            $result['privateKey'] = "-----BEGIN PRIVATE KEY-----\n" . $sections['private key'] . "\n-----END PRIVATE KEY-----";
        } elseif (isset($sections['rsa private key'])) {
            $result['privateKey'] = "-----BEGIN RSA PRIVATE KEY-----\n" . $sections['rsa private key'] . "\n-----END RSA PRIVATE KEY-----";
        }

        // Cadeia de certificados
        if (isset($sections['certificate chain'])) {
            $result['chain'] = "-----BEGIN CERTIFICATE-----\n" . $sections['certificate chain'] . "\n-----END CERTIFICATE-----";
        }

        return !empty($result['certificate']) && !empty($result['privateKey']) ? $result : false;
    }

    /**
     * Valida se o certificado está válido
     * 
     * @param Certificate $certificate
     * @return bool
     */
    public static function isValidCertificate($certificate)
    {
        try {
            $certData = openssl_x509_read($certificate->publicKey);
            if (!$certData) {
                return false;
            }

            $certInfo = openssl_x509_parse($certData);
            if (!$certInfo) {
                return false;
            }

            $currentTime = time();
            $validFrom = $certInfo['validFrom_time_t'];
            $validTo = $certInfo['validTo_time_t'];

            return ($currentTime >= $validFrom && $currentTime <= $validTo);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Obtém informações do certificado
     * 
     * @param Certificate $certificate
     * @return array
     */
    public static function getCertificateInfo($certificate)
    {
        try {
            $certData = openssl_x509_read($certificate->publicKey);
            if (!$certData) {
                return [];
            }

            $certInfo = openssl_x509_parse($certData);
            if (!$certInfo) {
                return [];
            }

            return [
                'subject' => $certInfo['subject'],
                'issuer' => $certInfo['issuer'],
                'validFrom' => date('Y-m-d H:i:s', $certInfo['validFrom_time_t']),
                'validTo' => date('Y-m-d H:i:s', $certInfo['validTo_time_t']),
                'serialNumber' => $certInfo['serialNumber'],
                'type' => self::detectCertificateType($certificate->publicKey, '')
            ];
        } catch (\Exception $e) {
            return [];
        }
    }
}

