<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class NPSPesquisa extends Model
{
    use HasFactory;
    
    protected $table = 'nps_pesquisas';
    
    protected $fillable = [
        'usuario_id',
        'empresa_id',
        'nota',
        'feedback',
        'ignorado',
        'respondido',
        'mes_referencia'
    ];
    
    protected $casts = [
        'ignorado' => 'boolean',
        'respondido' => 'boolean',
        'mes_referencia' => 'date'
    ];
    
    /**
     * Obtém o usuário relacionado à pesquisa
     */
    public function usuario()
    {
        return $this->belongsTo(User::class, 'usuario_id');
    }
    
    /**
     * Obtém a empresa relacionada à pesquisa
     */
    public function empresa()
    {
        return $this->belongsTo(Empresa::class, 'empresa_id');
    }
    
    /**
     * Verifica se um usuário já respondeu a pesquisa no mês atual
     * 
     * @param int $usuarioId ID do usuário
     * @return bool
     */
    public static function usuarioRespondeuNoMesAtual($usuarioId)
    {
        $primeiroDiaMes = Carbon::now()->startOfMonth();
        
        return self::where('usuario_id', $usuarioId)
            ->where('mes_referencia', $primeiroDiaMes->format('Y-m-d'))
            ->where('respondido', true)
            ->exists();
    }
    
    /**
     * Verifica se um usuário ignorou a pesquisa no mês atual
     * 
     * @param int $usuarioId ID do usuário
     * @return bool
     */
    public static function usuarioIgnorouNoMesAtual($usuarioId)
    {
        $primeiroDiaMes = Carbon::now()->startOfMonth();
        
        return self::where('usuario_id', $usuarioId)
            ->where('mes_referencia', $primeiroDiaMes->format('Y-m-d'))
            ->where('ignorado', true)
            ->exists();
    }
    
    /**
     * Cria ou atualiza o registro de pesquisa NPS para o usuário no mês atual
     * 
     * @param int $usuarioId ID do usuário
     * @param int $empresaId ID da empresa
     * @param array $dados Dados da pesquisa (nota, feedback, ignorado)
     * @return NPSPesquisa
     */
    public static function criarOuAtualizarPesquisa($usuarioId, $empresaId, $dados)
    {
        $primeiroDiaMes = Carbon::now()->startOfMonth();
        
        $pesquisa = self::firstOrNew([
            'usuario_id' => $usuarioId,
            'empresa_id' => $empresaId,
            'mes_referencia' => $primeiroDiaMes->format('Y-m-d')
        ]);
        
        // Preencher dados da pesquisa
        if (isset($dados['nota'])) {
            $pesquisa->nota = $dados['nota'];
            $pesquisa->respondido = true;
        }
        
        if (isset($dados['feedback'])) {
            $pesquisa->feedback = $dados['feedback'];
        }
        
        if (isset($dados['ignorado']) && $dados['ignorado']) {
            $pesquisa->ignorado = true;
        }
        
        $pesquisa->save();
        
        return $pesquisa;
    }
    
    /**
     * Classifica o respondente com base na nota NPS
     * 
     * @return string Classificação (detrator, neutro ou promotor)
     */
    public function getClassificacaoAttribute()
    {
        if ($this->nota <= 6) {
            return 'detrator';
        } elseif ($this->nota <= 8) {
            return 'neutro';
        } else {
            return 'promotor';
        }
    }
}
