<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\Nfce;
use App\Models\Nfe;
use App\Models\Empresa;
use App\Models\SangriaCaixa;
use App\Models\SuprimentoCaixa;
use App\Models\ItemContaEmpresa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Utils\ContaEmpresaUtil;
use Dompdf\Dompdf;
use Illuminate\Support\Facades\Log;

class SangriaController extends Controller
{

    protected $util;
    public function __construct(ContaEmpresaUtil $util){
        $this->util = $util;
    }

    public function store(Request $request)
    {
        try {
            // Log para debug
            Log::info('Sangria - Dados recebidos:', $request->all());
            
            // Garantir que o caixa_id está presente
            if (!$request->caixa_id) {
                // Tentar buscar o caixa aberto do usuário e empresa
                $caixa = \App\Models\Caixa::where('empresa_id', request()->empresa_id)
                    ->where('usuario_id', \Auth::id())
                    ->where('status', 1)
                    ->first();
                if ($caixa) {
                    $request->merge(['caixa_id' => $caixa->id]);
                } else {
                    session()->flash('flash_error', 'Erro: Nenhum caixa aberto para registrar a sangria!');
                    return redirect()->back();
                }
            }

            // Validação dos dados recebidos
            if(!$request->valor || __convert_value_bd($request->valor) == 0){
                Log::warning('Sangria - Valor inválido:', ['valor' => $request->valor]);
                session()->flash("flash_error", "Informe um valor maior que zero");
                return redirect()->back();
            }
            
            // Verificar se o caixa existe
            $caixa = Caixa::find($request->caixa_id);
            if (!$caixa) {
                Log::error('Sangria - Caixa não encontrado:', ['caixa_id' => $request->caixa_id]);
                session()->flash("flash_error", "Caixa não encontrado");
                return redirect()->back();
            }
            
            // Registra a sangria no banco de dados
            $sangria = SangriaCaixa::create([
                'caixa_id' => $request->caixa_id,
                'valor' => __convert_value_bd($request->valor),
                'observacao' => $request->observacao ?? '',
            ]);
            
            Log::info('Sangria - Criada com sucesso:', ['sangria_id' => $sangria->id, 'valor' => $sangria->valor]);
            
            // Processa as transações de conta empresa, se necessário
            if($request->conta_empresa_sangria_id){
                $caixa = Caixa::findOrFail($request->caixa_id);
                $data = [
                    'conta_id' => $caixa->conta_empresa_id,
                    'descricao' => "Sangria de caixa",
                    'tipo_pagamento' => '01',
                    'valor' => __convert_value_bd($request->valor),
                    'caixa_id' => $caixa->id,
                    'tipo' => 'saida'
                ];
                $itemContaEmpresa = ItemContaEmpresa::create($data);
                $this->util->atualizaSaldo($itemContaEmpresa);

                $data = [
                    'conta_id' => $request->conta_empresa_sangria_id,
                    'descricao' => "Sangria de caixa",
                    'tipo_pagamento' => '01',   
                    'valor' => __convert_value_bd($request->valor),
                    'caixa_id' => $caixa->id,
                    'tipo' => 'entrada'
                ];
                $itemContaEmpresa = ItemContaEmpresa::create($data);
                $this->util->atualizaSaldo($itemContaEmpresa);
            }
            
            // Armazena o ID na sessão para abrir a impressão após o redirecionamento
            session()->flash('sangria_id', $sangria->id);
            session()->flash('flash_success', 'Sangria realizada com sucesso!');
            
            Log::info('Sangria - Finalizada com sucesso', [
                'sangria_id' => $sangria->id,
                'caixa_id' => $sangria->caixa_id,
                'valor' => $sangria->valor,
                'observacao' => $sangria->observacao
            ]);
            
            return redirect()->back();
            
        } catch (\Exception $e) {
            // Registra o erro no log
            Log::error("Erro ao processar sangria: " . $e->getMessage());
            
            // Feedback para o usuário
            session()->flash("flash_error", "Erro ao processar sangria: " . $e->getMessage());
            return redirect()->back();
        }
    }

    public function print($id){
        $sangria = SangriaCaixa::findOrfail($id);
        $empresa = Empresa::findOrFail(request()->empresa_id);
        
        $p = view('front_box.sangria_print', compact('sangria', 'empresa'));
        $domPdf = new Dompdf(["enable_remote" => true]);
        $domPdf->loadHtml($p);

        $pdf = ob_get_clean();

        $domPdf->set_paper(array(0,0,214,220));
        $domPdf->render();
        $domPdf->stream("Comprovante de sangria.pdf", array("Attachment" => false));
        exit();
    }

    private function somaTotalEmCaixa()
    {
        $usuario_id = Auth::user()->id;
        $abertura = Caixa::where('empresa_id', request()->empresa_id)->where('status', 1)->where('usuario_id', $usuario_id)
        ->orderBy('id', 'desc')
        ->first();
        // dd($abertura->valor_abertura);
        if ($abertura == null) return 0;
        $soma = 0;
        $soma += $abertura->valor_abertura;
        $nfce = Nfce::selectRaw('sum(total) as valor')->where('empresa_id', request()->empresa_id)->where('caixa_id', $abertura->id)
        ->first();
        if ($nfce != null)
            $soma += $nfce->valor;
        $nfe = Nfe::selectRaw('sum(total) as valor')->where('empresa_id', request()->empresa_id)->where('caixa_id', $abertura->id)
        ->first();
        if ($nfe != null)
            $soma += $nfe->valor;
        $amanha = date('Y-m-d', strtotime('+1 days')) . " 00:00:00";
        $suprimentosSoma = SuprimentoCaixa::selectRaw('sum(valor) as valor')->whereBetween('created_at', [$abertura->created_at, $amanha])
        ->where('caixa_id', $abertura->id)
        ->first();
        if ($suprimentosSoma != null)
            $soma += $suprimentosSoma->valor;
        $sangriasSoma = SangriaCaixa::selectRaw('sum(valor) as valor')->whereBetween('created_at', [$abertura->created_at, $amanha])
        ->where('caixa_id', $abertura->id)
        ->first();
        if ($sangriasSoma != null)
            $soma -= $sangriasSoma->valor;
        return $soma;
    }
}
