<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\RemessaBoleto;
use App\Models\RemessaBoletoItem;
use App\Models\ContaBoleto;
use App\Models\ContaReceber;
use App\Models\Boleto;
use App\Models\Cliente;
use App\Utils\BoletoUtil;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class RemessaBoletoController extends Controller
{

    protected $util;

    public function __construct(BoletoUtil $util)
    {
        $this->util = $util;
        try {
            $path = public_path('remessas_boleto');
            $path = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $path);
            
            if (!file_exists($path)) {
                if (!mkdir($path, 0777, true)) {
                    throw new \Exception("Não foi possível criar o diretório: " . $path);
                }
                chmod($path, 0777);
            }
        } catch (\Exception $e) {
            Log::error('Erro ao criar diretório de remessas: ' . $e->getMessage());
        }
    } 

    private function getRemessaPath($empresa_id)
    {
        try {
            // Recupera os dados da empresa
            $empresa = \App\Models\Empresa::findOrFail($empresa_id);
        
            // Formata o CNPJ, removendo pontos, barras e traços
            $cnpj = preg_replace('/[^0-9]/', '', $empresa->cpf_cnpj);
        
            // Define o caminho base
            $basePath = public_path('remessas_boleto');
            
            // Normaliza o caminho base
            $basePath = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $basePath);
        
            // Monta o caminho completo com separador adequado
            $path = $basePath . DIRECTORY_SEPARATOR . $cnpj;
            
            // Normaliza o caminho completo
            $path = str_replace(['\\', '/'], DIRECTORY_SEPARATOR, $path);
        
            // Verifica se o diretório base existe, se não, cria
            if (!File::exists($basePath)) {
                File::makeDirectory($basePath, 0777, true);
                chmod($basePath, 0777);
            }
        
            // Verifica se o diretório da empresa existe, se não, cria
            if (!File::exists($path)) {
                File::makeDirectory($path, 0777, true);
                chmod($path, 0777);
            }
        
            Log::info('Caminho da remessa criado: ' . $path);
            return $path;
            
        } catch (\Exception $e) {
            Log::error('Erro ao criar diretório da remessa: ' . $e->getMessage());
            throw new \Exception('Erro ao criar diretório da remessa: ' . $e->getMessage());
        }
    }
        

    public function index(Request $request){

        $contasBoleto = ContaBoleto::where('empresa_id', $request->empresa_id)
        ->get();

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $conta_boleto_id = $request->get('conta_boleto_id');

        $data = RemessaBoleto::where('empresa_id', $request->empresa_id)
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('created_at', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date,) {
            return $query->whereDate('created_at', '<=', $end_date);
        })
        ->when(!empty($conta_boleto_id), function ($query) use ($conta_boleto_id) {
            return $query->where('conta_boleto_id', $conta_boleto_id);
        })
        ->orderBy('created_at', 'desc')
        ->paginate(env("PAGINACAO"));

        return view('remessa_boletos.index', compact('data', 'contasBoleto'));
    }

    public function create(Request $request){
        $contasBoleto = ContaBoleto::where('empresa_id', request()->empresa_id)
        ->get();

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $conta_boleto_id = $request->get('conta_boleto_id');

        $boletos = RemessaBoletoItem::join('boletos', 'boletos.id', '=', 'remessa_boleto_items.boleto_id')
        ->where('boletos.empresa_id', request()->empresa_id)
        ->pluck('remessa_boleto_items.boleto_id')
        ->all();

        $data = Boleto::where('empresa_id', request()->empresa_id)
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('vencimento', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date,) {
            return $query->whereDate('vencimento', '<=', $end_date);
        })->when(!empty($conta_boleto_id), function ($query) use ($conta_boleto_id) {
            return $query->where('conta_boleto_id', $conta_boleto_id);
        })
        ->orderBy('id', 'desc')
        ->whereNotIn('id', $boletos)->get();

        return view('remessa_boletos.create', compact('data', 'contasBoleto'));

    }

    public function store(Request $request){
        if(!$request->boleto_id){
            session()->flash('flash_error', 'Selecione ao menos um boleto para gerar');
            return redirect()->back();
        }
        try{
            $tipo = null;
            $banco = null;
            DB::transaction(function () use ($request) {
                $conta = null;
                $boleto = Boleto::findOrFail($request->boleto_id[0]);
                $tipo = $boleto->tipo;
                $contaBoleto = ContaBoleto::findOrFail($boleto->conta_boleto_id);
                $beneficiario = new \Eduardokum\LaravelBoleto\Pessoa([
                    'nome' => $contaBoleto->titular,
                    'endereco' => "$contaBoleto->rua, $contaBoleto->numero",
                    'bairro' => $contaBoleto->bairro,
                    'cep' => $contaBoleto->cep,
                    'uf' => $contaBoleto->cidade->uf,
                    'cidade' => $contaBoleto->cidade->nome,
                    'documento' => $contaBoleto->documento,
                ]);

                $boletos = [];
                for($i=0; $i<sizeof($request->boleto_id); $i++){
                    $boleto = Boleto::findOrFail($request->boleto_id[$i]);
                    if($conta == null){
                        $conta = $boleto->conta_boleto_id;
                    }

                    if($conta != $boleto->conta_boleto_id){
                        session()->flash('flash_error', 'Selecione um ou mais boletos do mesmo banco!');
                        return redirect()->back();
                    }

                    $contaReceber = $boleto->contaReceber;

                    $pagador = new \Eduardokum\LaravelBoleto\Pessoa([
                        'nome' => $contaReceber->cliente->razao_social,
                        'endereco' => $contaReceber->cliente->rua . ", " . $contaReceber->cliente->numero,
                        'bairro' => $contaReceber->cliente->bairro,
                        'cep' => $contaReceber->cliente->cep,
                        'uf' => $contaReceber->cliente->cidade->uf,
                        'cidade' => $contaReceber->cliente->cidade->nome,
                        'documento' => $contaReceber->cliente->cpf_cnpj,
                    ]);

                    $dataBoleto = [
                        'conta_boleto_id' => $boleto->conta_boleto_id,
                        'numero' => $boleto->numero,
                        'numero_documento' => $boleto->numero_documento,
                        'vencimento' => $boleto->vencimento,
                        'valor' => $boleto->valor,
                        'carteira' => $boleto->carteira,
                        'convenio' => $boleto->convenio,
                        'juros' => $boleto->juros,
                        'multa' => $boleto->multa,
                        'juros_apos' => $boleto->juros_apos,
                        'instrucoes' => $boleto->instrucoes ?? '', 
                        'usar_logo' => $boleto->usar_logo,
                        'tipo' => $request->tipo,
                        'codigo_cliente' => null,
                        'posto' => null,
                        'empresa_id' => $boleto->empresa_id,
                        'cliente_id' => $contaReceber->cliente_id
                    ];

                    $boletos = DB::transaction(function () use ($beneficiario, $pagador, $dataBoleto, $contaBoleto, $boletos) {
                        $boleto = $this->util->gerarBoletoParaRemessa($beneficiario, $pagador, $dataBoleto, $contaBoleto);
                        if($boleto){
                            array_push($boletos, $boleto);
                        }
                        return $boletos;
                    });
                }

                $sequencial = $contaBoleto->banco == 'Sicredi' ? $contaBoleto->sequencial_remessa : null;

                $dataRemessa = [
                    'agencia' => $contaBoleto->agencia,
                    'carteira' => $boleto->carteira,
                    'conta' => $contaBoleto->conta,
                    'convenio' => $boleto->convenio,
                    'variacaoCarteira' => '1',
                    'idremessa' => $contaBoleto->banco == 'Sicredi' ? $sequencial : Str::random(30),
                    'beneficiario' => $beneficiario,
                    'codigoCliente' => $contaReceber->cliente_id
                ];

                // Gera o nome do arquivo com data e hora
                $nomeArquivo = $contaBoleto->banco == 'Sicredi' ? 
                    date('dmHi') . '.REM' : 
                    date('d_m_Y_H_i_s') . '_' . Str::random(10) . '.txt';
                
                // Define o caminho do arquivo na pasta da empresa
                $path = $this->getRemessaPath($request->empresa_id);
                
                // Gera a remessa
                $this->util->geraRemessa($boletos, $tipo, $contaBoleto->banco, $dataRemessa, $nomeArquivo, $path);
                
                // Monta o caminho completo do arquivo
                $filePath = rtrim($path, DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . $nomeArquivo;
                
                // Se for Sicredi, atualiza o arquivo com o sequencial formatado
                if($contaBoleto->banco == 'Sicredi') {
                    $filePath = preg_replace('/[[:^print:]]/', '', $filePath); // limpeza
                    $realPath = realpath($filePath);
                    if (!$realPath || !file_exists($realPath)) {
                        throw new \Exception("Arquivo da remessa não encontrado em: " . $filePath);
                    }
                
                    $conteudo = file_get_contents($filePath);
                
                    // Formata o sequencial com 7 dígitos no header (posição 111-117)
                    $sequencialFormatado = str_pad($sequencial, 7, '0', STR_PAD_LEFT);
                    $conteudo = substr_replace($conteudo, $sequencialFormatado, 110, 7);
                
                    file_put_contents($filePath, $conteudo);
                }

                $remessaBoleto = RemessaBoleto::create([
                    'nome_arquivo' => $nomeArquivo,
                    'empresa_id' => $request->empresa_id,
                    'conta_boleto_id' => $contaBoleto->id
                ]);

                for($i=0; $i<sizeof($request->boleto_id); $i++){
                    RemessaBoletoItem::create([
                        'remessa_id' => $remessaBoleto->id,
                        'boleto_id' => $request->boleto_id[$i]
                    ]);
                }

                // Incrementa o sequencial da remessa após gerar o arquivo
                if($contaBoleto->banco == 'Sicredi') {
                    $contaBoleto->sequencial_remessa = $contaBoleto->sequencial_remessa + 1;
                    $contaBoleto->save();
                }
            });

            session()->flash("flash_success", 'Remessa gerada!');
            return redirect()->route('remessa-boleto.index');
        }catch(\Exception $e){
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
            return redirect()->back();
        }
    }

    public function download($id){
        $item = RemessaBoleto::findOrFail($id);
        
        // Busca o arquivo na pasta da empresa
        $path = $this->getRemessaPath($item->empresa_id);
        
        // Monta o caminho completo do arquivo
        $filePath = implode(DIRECTORY_SEPARATOR, [$path, $item->nome_arquivo]);
        
        // Normaliza o caminho
        $filePath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $filePath);
        
        if(!file_exists($filePath)){
            session()->flash("flash_error", 'Arquivo não existe');
            return redirect()->back();
        }

        return response()->download($filePath);
    }

    public function destroy($id){
        $item = RemessaBoleto::findOrFail($id);
        
        // Remove o arquivo da pasta da empresa
        $path = $this->getRemessaPath($item->empresa_id);
        
        // Monta o caminho completo do arquivo
        $filePath = implode(DIRECTORY_SEPARATOR, [$path, $item->nome_arquivo]);
        
        // Normaliza o caminho
        $filePath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $filePath);
        
        if(file_exists($filePath)){
            unlink($filePath);
        }
        $item->itens()->delete();
        $item->delete();
        session()->flash("flash_success", 'Remessa removida!');

        return redirect()->back();
    }

    public function import(Request $request){
        $contasBoleto = ContaBoleto::where('empresa_id', $request->empresa_id)
        ->get();

        return view('remessa_boletos.import', compact('contasBoleto'));
    }

    public function importStore(Request $request){
        if ($request->hasFile('file')) {
            $retorno = \Eduardokum\LaravelBoleto\Cnab\Retorno\Factory::make($request->file);
            $retorno->processar();

            $banco = $retorno->getBancoNome();
            $data = [];

            $contasPendentes = ContaReceber::where('empresa_id', $request->empresa_id)
            ->get();

            foreach($retorno->getDetalhes() as $item){
                $r = [
                    'ocorrencia' => $item->ocorrenciaDescricao,
                    'carteira' => $item->carteira,
                    'valor_integral' => $item->valor,
                    'valor_recebido' => $item->valorRecebido,
                    'valor_tarifa' => $item->valorTarifa,
                    'vencimento' => $item->dataVencimento,
                    'pagador' => $item->getPagador()->getNome(),
                    'documento' => $item->getPagador()->getDocumento(),
                    'conta_id' => null
                ];
                $conta = $this->getContaReceber((object)$r);
                if($conta != null){
                    $r['conta_id'] = $conta->id;
                }
                array_push($data, (object)$r);

            }
            // dd($data);
            return view('remessa_boletos.import_view', compact('data', 'banco', 'contasPendentes'));
        } else {
            session()->flash('flash_error', 'Nenhum Arquivo!!');
            return redirect()->back();
        }
    }

    private function getContaReceber($r){

        $cliente = Cliente::where('empresa_id', request()->empresa_id)
        ->where('cpf_cnpj', $r->documento)->first();
        if($cliente){
            // dd($r);
            $vencimento = str_replace("/", "-", $r->vencimento);
            $conta = ContaReceber::where('cliente_id', $cliente->id)
            ->where('valor_integral', $r->valor_integral)
            ->whereDate('data_vencimento', \Carbon\Carbon::parse($vencimento)->format('Y-m-d'))
            ->first();
            return $conta;
        }
        return null;
    }

    public function importSave(Request $request){
        try{
            $cont = 0;
            for($i=0; $i<sizeof($request->conta_id); $i++){
                $conta = ContaReceber::findOrFail($request->conta_id[$i]);
                if(!$conta->status){
                    $cont++;
                }
                $conta->valor_recebido = __convert_value_bd($request->valor_recebido[$i]);
                $conta->data_recebimento = $request->data_recebimento[$i];
                $conta->status = true;
                $conta->save();

            }
            session()->flash("flash_success", 'Total de contas recebidas: ' . $cont);
            return redirect()->route('conta-receber.index');    
        }catch(\Exception $e){
                // echo $e->getMessage();
                // die;
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
            return redirect()->back();
        }
    }

}
