<?php

namespace App\Http\Controllers;

use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\TaxaPagamento;
use Illuminate\Http\Request;
use Dompdf\Dompdf;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardTaxaPagamentoController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:taxa_pagamento_view');
    }

    public function index(Request $request)
    {
        $dataInicial = $request->data_inicial ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $dataFinal = $request->data_final ?? Carbon::now()->format('Y-m-d');

        // Obtém as taxas cadastradas
        $taxas = TaxaPagamento::where('empresa_id', request()->empresa_id)->get();
        $tipos = $taxas->pluck('tipo_pagamento')->toArray();

        // Obtém dados para o dashboard
        $dadosDashboard = $this->getDadosDashboard($dataInicial, $dataFinal, $tipos);

        // Prepara dados para os gráficos
        $dadosGraficos = $this->prepararDadosGraficos($dadosDashboard);

        return view('dashboard_taxa_pagamento.index', compact(
            'dadosDashboard', 
            'dadosGraficos', 
            'dataInicial', 
            'dataFinal'
        ));
    }

    public function relatorio(Request $request)
    {
        $dataInicial = $request->data_inicial;
        $dataFinal = $request->data_final;

        // Obtém as taxas cadastradas
        $taxas = TaxaPagamento::where('empresa_id', request()->empresa_id)->get();
        $tipos = $taxas->pluck('tipo_pagamento')->toArray();

        // Obtém dados para o relatório
        $dados = $this->getDadosDashboard($dataInicial, $dataFinal, $tipos);

        // Preparar o relatório PDF
        $p = view('dashboard_taxa_pagamento.relatorio')
            ->with('dados', $dados)
            ->with('dataInicial', $dataInicial)
            ->with('dataFinal', $dataFinal)
            ->with('title', 'Relatório de Taxas de Pagamento');

        $domPdf = new Dompdf(["enable_remote" => true]);
        $domPdf->loadHtml($p);

        $pdf = ob_get_clean();

        $domPdf->setPaper("A4", "landscape");
        $domPdf->render();
        $domPdf->stream("Relatório de Taxas de Pagamento.pdf", array("Attachment" => false));
    }

    private function getDadosDashboard($dataInicial, $dataFinal, $tipos)
    {
        $dadosVendas = [];
        $totalVendas = 0;
        $totalTaxas = 0;
        $vendasSemTaxa = 0;
        
        // Processa NFes
        $vendas = Nfe::where('empresa_id', request()->empresa_id)
            ->where('orcamento', 0) // Somente vendas confirmadas
            ->whereDate('created_at', '>=', $dataInicial)
            ->whereDate('created_at', '<=', $dataFinal)
            ->get();

        foreach ($vendas as $venda) {
            $bandeira_cartao = $venda->bandeira_cartao;
            
            // Verifica se a venda tem múltiplas formas de pagamento
            if (sizeof($venda->fatura) > 1) {
                foreach ($venda->fatura as $fatura) {
                    $fp = $fatura->tipo_pagamento;
                    if (in_array($fp, $tipos)) {
                        // Buscar apenas para obter o nome da bandeira
                        $bandeiraNome = '--'; // Valor padrão
                        $taxa = TaxaPagamento::where('empresa_id', request()->empresa_id)
                            ->where('tipo_pagamento', $fp)
                            ->when($bandeira_cartao != '' && $bandeira_cartao != '99', function ($q) use ($bandeira_cartao) {
                                return $q->where('bandeira_cartao', $bandeira_cartao);
                            })
                            ->first();
                            
                        if ($taxa !== null && $taxa->bandeira_cartao) {
                            $bandeiraNome = $taxa->getBandeira();
                        }
                        
                        // Usar a taxa salva na fatura, se disponível, senão usar zero
                        $taxaPercentual = $fatura->taxa_percentual ?? 0;
                        if ($fatura->taxa_percentual === null) {
                            $vendasSemTaxa++;
                        }
                        
                        $valorTaxa = $fatura->valor * ($taxaPercentual / 100);
                        $dadosVendas[] = [
                            'cliente' => $venda->cliente ? ($venda->cliente->razao_social . " " . $venda->cliente->cpf_cnpj) : 'Consumidor final',
                            'total' => $fatura->valor,
                            'taxa_perc' => $taxaPercentual,
                            'valor_taxa' => $valorTaxa,
                            'valor_liquido' => $fatura->valor - $valorTaxa,
                            'data' => Carbon::parse($venda->created_at)->format('d/m/Y H:i'),
                            'tipo_pagamento' => Nfe::getTipo($fp),
                            'codigo_pagamento' => $fp,
                            'bandeira' => $bandeiraNome,
                            'venda_id' => $venda->id,
                            'tipo' => 'PEDIDO',
                            'taxa_historica' => ($fatura->taxa_percentual !== null)
                        ];
                        $totalVendas += $fatura->valor;
                        $totalTaxas += $valorTaxa;
                    }
                }
            } else {
                if (in_array($venda->tipo_pagamento, $tipos)) {
                    $total = $venda->valor_total - $venda->desconto + $venda->acrescimo;
                    
                    // Buscar apenas para obter o nome da bandeira
                    $bandeiraNome = '--'; // Valor padrão
                    $taxa = TaxaPagamento::where('empresa_id', request()->empresa_id)
                        ->where('tipo_pagamento', $venda->tipo_pagamento)
                        ->when($bandeira_cartao != '' && $bandeira_cartao != '99', function ($q) use ($bandeira_cartao) {
                            return $q->where('bandeira_cartao', $bandeira_cartao);
                        })
                        ->first();
                        
                    if ($taxa !== null && $taxa->bandeira_cartao) {
                        $bandeiraNome = $taxa->getBandeira();
                    }
                    
                    // Buscar a fatura associada à venda para obter a taxa
                    $fatura = $venda->fatura->first();
                    // Usar a taxa salva na fatura, se disponível, senão usar zero
                    $taxaPercentual = $fatura ? ($fatura->taxa_percentual ?? 0) : 0;
                    if (!$fatura || $fatura->taxa_percentual === null) {
                        $vendasSemTaxa++;
                    }
                    
                    $valorTaxa = $total * ($taxaPercentual / 100);
                    $dadosVendas[] = [
                        'cliente' => $venda->cliente ? ($venda->cliente->razao_social . " " . $venda->cliente->cpf_cnpj) : 'Consumidor final',
                        'total' => $total,
                        'taxa_perc' => $taxaPercentual,
                        'valor_taxa' => $valorTaxa,
                        'valor_liquido' => $total - $valorTaxa,
                        'data' => Carbon::parse($venda->created_at)->format('d/m/Y H:i'),
                        'tipo_pagamento' => Nfe::getTipo($venda->tipo_pagamento),
                        'codigo_pagamento' => $venda->tipo_pagamento,
                        'bandeira' => $bandeiraNome,
                        'venda_id' => $venda->id,
                        'tipo' => 'PEDIDO',
                        'taxa_historica' => ($fatura && $fatura->taxa_percentual !== null)
                    ];
                    $totalVendas += $total;
                    $totalTaxas += $valorTaxa;
                }
            }
        }

        // Processa NFCes (PDV)
        $vendasCaixa = Nfce::where('empresa_id', request()->empresa_id)
            ->whereDate('created_at', '>=', $dataInicial)
            ->whereDate('created_at', '<=', $dataFinal)
            ->get();

        foreach ($vendasCaixa as $venda) {
            $bandeira_cartao = $venda->bandeira_cartao;
            
            // Verifica se a venda tem múltiplas formas de pagamento
            if (sizeof($venda->fatura) > 1) {
                foreach ($venda->fatura as $fatura) {
                    $fp = $fatura->tipo_pagamento;
                    if (in_array($fp, $tipos)) {
                        // Buscar apenas para obter o nome da bandeira
                        $bandeiraNome = '--'; // Valor padrão
                        $taxa = TaxaPagamento::where('empresa_id', request()->empresa_id)
                            ->where('tipo_pagamento', $fp)
                            ->when($bandeira_cartao != '' && $bandeira_cartao != '99', function ($q) use ($bandeira_cartao) {
                                return $q->where('bandeira_cartao', $bandeira_cartao);
                            })
                            ->first();
                            
                        if ($taxa !== null && $taxa->bandeira_cartao) {
                            $bandeiraNome = $taxa->getBandeira();
                        }
                        
                        // Usar a taxa salva na fatura, se disponível, senão usar zero
                        $taxaPercentual = $fatura->taxa_percentual ?? 0;
                        if ($fatura->taxa_percentual === null) {
                            $vendasSemTaxa++;
                        }
                        
                        $valorTaxa = $fatura->valor * ($taxaPercentual / 100);
                        $dadosVendas[] = [
                            'cliente' => $venda->cliente ? ($venda->cliente->razao_social . " " . $venda->cliente->cpf_cnpj) : 'Consumidor final',
                            'total' => $fatura->valor,
                            'taxa_perc' => $taxaPercentual,
                            'valor_taxa' => $valorTaxa,
                            'valor_liquido' => $fatura->valor - $valorTaxa,
                            'data' => Carbon::parse($venda->created_at)->format('d/m/Y H:i'),
                            'tipo_pagamento' => Nfce::getTipoPagamento($fp),
                            'codigo_pagamento' => $fp,
                            'bandeira' => $bandeiraNome,
                            'venda_id' => $venda->id,
                            'tipo' => 'PDV',
                            'taxa_historica' => ($fatura->taxa_percentual !== null)
                        ];
                        $totalVendas += $fatura->valor;
                        $totalTaxas += $valorTaxa;
                    }
                }
            } else {
                if (in_array($venda->tipo_pagamento, $tipos)) {
                    // Buscar apenas para obter o nome da bandeira
                    $bandeiraNome = '--'; // Valor padrão
                    $taxa = TaxaPagamento::where('empresa_id', request()->empresa_id)
                        ->where('tipo_pagamento', $venda->tipo_pagamento)
                        ->when($bandeira_cartao != '' && $bandeira_cartao != '99', function ($q) use ($bandeira_cartao) {
                            return $q->where('bandeira_cartao', $bandeira_cartao);
                        })
                        ->first();
                        
                    if ($taxa !== null && $taxa->bandeira_cartao) {
                        $bandeiraNome = $taxa->getBandeira();
                    }
                    
                    // Buscar a fatura associada à venda para obter a taxa
                    $fatura = $venda->fatura->first();
                    // Usar a taxa salva na fatura, se disponível, senão usar zero
                    $taxaPercentual = $fatura ? ($fatura->taxa_percentual ?? 0) : 0;
                    if (!$fatura || $fatura->taxa_percentual === null) {
                        $vendasSemTaxa++;
                    }
                    
                    $valorTaxa = $venda->total * ($taxaPercentual / 100);
                    $dadosVendas[] = [
                        'cliente' => $venda->cliente ? ($venda->cliente->razao_social . " " . $venda->cliente->cpf_cnpj) : 'Consumidor final',
                        'total' => $venda->total,
                        'taxa_perc' => $taxaPercentual,
                        'valor_taxa' => $valorTaxa,
                        'valor_liquido' => $venda->total - $valorTaxa,
                        'data' => Carbon::parse($venda->created_at)->format('d/m/Y H:i'),
                        'tipo_pagamento' => Nfce::getTipoPagamento($venda->tipo_pagamento),
                        'codigo_pagamento' => $venda->tipo_pagamento,
                        'bandeira' => $bandeiraNome,
                        'venda_id' => $venda->id,
                        'tipo' => 'PDV',
                        'taxa_historica' => ($fatura && $fatura->taxa_percentual !== null)
                    ];
                    $totalVendas += $venda->total;
                    $totalTaxas += $valorTaxa;
                }
            }
        }

        return [
            'vendas' => $dadosVendas,
            'totalVendas' => $totalVendas,
            'totalTaxas' => $totalTaxas,
            'totalLiquido' => $totalVendas - $totalTaxas,
            'vendasSemTaxa' => $vendasSemTaxa
        ];
    }

    private function prepararDadosGraficos($dadosDashboard)
    {
        $dados = $dadosDashboard['vendas'];
        
        // Dados para gráfico por forma de pagamento
        $porFormaPagamento = [];
        foreach ($dados as $venda) {
            // Incluir apenas vendas com taxa real registrada (não as com taxa zero)
            if ($venda['taxa_historica']) {
                $tipo = $venda['tipo_pagamento'];
                
                if (!isset($porFormaPagamento[$tipo])) {
                    $porFormaPagamento[$tipo] = [
                        'total' => 0,
                        'taxas' => 0,
                        'codigo' => $venda['codigo_pagamento']
                    ];
                }
                
                $porFormaPagamento[$tipo]['total'] += $venda['total'];
                $porFormaPagamento[$tipo]['taxas'] += $venda['valor_taxa'];
            }
        }
        
        // Dados para gráfico por bandeira de cartão
        $porBandeira = [];
        foreach ($dados as $venda) {
            // Incluir apenas vendas com taxa real registrada (não as com taxa zero)
            if ($venda['taxa_historica'] && $venda['bandeira'] != '--') {
                $bandeira = $venda['bandeira'];
                
                if (!isset($porBandeira[$bandeira])) {
                    $porBandeira[$bandeira] = [
                        'total' => 0,
                        'taxas' => 0
                    ];
                }
                
                $porBandeira[$bandeira]['total'] += $venda['total'];
                $porBandeira[$bandeira]['taxas'] += $venda['valor_taxa'];
            }
        }
        
        // Dados para gráfico de evolução diária
        $evolucaoDiaria = [];
        foreach ($dados as $venda) {
            $data = substr($venda['data'], 0, 10); // Pega só a data, sem a hora
            
            if (!isset($evolucaoDiaria[$data])) {
                $evolucaoDiaria[$data] = [
                    'total' => 0,
                    'taxas' => 0
                ];
            }
            
            // Adicionar o total da venda sempre, mas adicionar taxas apenas se tiver taxa real
            $evolucaoDiaria[$data]['total'] += $venda['total'];
            if ($venda['taxa_historica']) {
                $evolucaoDiaria[$data]['taxas'] += $venda['valor_taxa'];
            }
        }
        
        // Ordenar evolução diária por data
        ksort($evolucaoDiaria);
        
        return [
            'porFormaPagamento' => $porFormaPagamento,
            'porBandeira' => $porBandeira,
            'evolucaoDiaria' => $evolucaoDiaria
        ];
    }
} 