<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Cliente;
use App\Models\ContaReceber;
use App\Models\CashBackConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ClienteController extends Controller
{
    public function find($id)
    {
        $item = Cliente::with(['cidade', 'listaPreco'])->findOrFail($id);
        return response()->json($item, 200);
    }

    public function cashback($id)
    {
        $item = Cliente::with('cidade')->findOrFail($id);
        $config = CashBackConfig::where('empresa_id', $item->empresa_id)->first();
        if($config == null){
            return response()->json(null, 404);
        }
        $config->valor_cashback = $item->valor_cashback;
        return response()->json($config, 200);
    }

    public function pesquisa(Request $request)
    {
        $data = Cliente::orderBy('razao_social', 'desc')
        ->where('empresa_id', $request->empresa_id)
        ->where('status', 1)
        ->where('razao_social', 'like', "%$request->pesquisa%")
        ->get();
        return response()->json($data, 200);
    }

    public function pesquisaDelivery(Request $request)
    {
        $data = Cliente::orderBy('razao_social', 'desc')
        ->where('empresa_id', $request->empresa_id)
        ->where('status', 1)
        ->where('uid', '!=', '')
        ->when(!is_numeric($request->pesquisa), function ($q) use ($request) {
            return $q->where('razao_social', 'LIKE', "%$request->pesquisa%");
        })
        ->when(is_numeric($request->pesquisa), function ($q) use ($request) {
            return $q->where('telefone', 'LIKE', "%$request->pesquisa%");
        })
        ->get();
        return response()->json($data, 200);
    }

    /**
     * Cria um novo cliente
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            $cliente = Cliente::where('empresa_id', $request->empresa_id)
                ->where('cpf_cnpj', $request->cpf_cnpj)
                ->first();
                
            if ($cliente) {
                return response()->json("Cliente já cadastrado", 401);
            }
            
            $cliente = Cliente::create($request->all());
            return response()->json($cliente, 200);
        } catch(\Exception $e) {
            return response()->json($e->getMessage(), 500);
        }
    }
    
    /**
     * Consulta débitos do cliente
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function consultaDebitos(Request $request)
    {
        $totalVenda = $request->total;
        $somaContas = ContaReceber::where('cliente_id', $request->cliente_id)
        ->where('status', 0)
        ->sum('valor_integral');

        $cliente = Cliente::findOrFail($request->cliente_id);
        
        // Verifica qual campo de crédito usar
        $limiteCredito = 0;
        if(isset($cliente->limite_credito) && $cliente->limite_credito > 0){
            $limiteCredito = $cliente->limite_credito;
        } else if(isset($cliente->valor_credito) && $cliente->valor_credito > 0){
            $limiteCredito = $cliente->valor_credito;
        }
        
        if($somaContas+$totalVenda > $limiteCredito){
            return response()->json("Valor ultrapassa o limite definido no cadastro do cliente", 403);
        }
        return response()->json($limiteCredito, 200);
    }
    
    /**
     * Valida limite de crédito do cliente
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validaLimiteCredito(Request $request)
    {
        try {
            // Implementação simplificada
            return response()->json([
                'success' => true,
                'message' => 'Validação de limite de crédito',
                'data' => [
                    'limite_valido' => true
                ]
            ], 200);
        } catch(\Exception $e) {
            return response()->json($e->getMessage(), 500);
        }
    }

    /**
     * Busca clientes por nome para o select2 no PDV
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function findByName(Request $request)
    {
        try {
            // Log para debug - registrar os dados da requisição
            Log::info('Requisição de busca de clientes: ', [
                'method' => $request->method(),
                'params' => $request->all(),
                'ip' => $request->ip(),
                'user_agent' => $request->header('User-Agent')
            ]);
            
            // Verificar se os parâmetros existem, com valores padrão seguros
            $query = $request->input('q', '');
            $empresa_id = $request->input('empresa_id', 0);
            
            // Log para debug - parâmetros processados
            Log::info('Parâmetros processados: ', [
                'query' => $query,
                'empresa_id' => $empresa_id
            ]);
            
            // Para fins de diagnóstico, vamos retornar um cliente vazio se nenhum for encontrado
            // em vez de falhar com erros de validação
            if (strlen($query) < 2 || empty($empresa_id)) {
                Log::warning('Parâmetros inválidos para busca de clientes', [
                    'query' => $query,
                    'empresa_id' => $empresa_id
                ]);
                
                // Retornar uma resposta vazia em vez de erro para diagnóstico
                return response()->json([], 200);
            }
            
            // Verificar se o modelo Cliente existe e tem os métodos necessários
            if (!class_exists('App\\Models\\Cliente')) {
                Log::error('Modelo Cliente não encontrado');
                return response()->json([
                    'error' => 'Configuração do sistema incorreta'
                ], 500);
            }
            
            // Realizar a consulta de forma segura com tratamento de exceções específicas
            try {
                $clientes = Cliente::where('empresa_id', $empresa_id)
                    ->where('status', 1)
                    ->where('razao_social', 'like', "%{$query}%")
                    ->select('id', 'razao_social', 'cpf_cnpj', 'telefone')
                    ->orderBy('razao_social')
                    ->limit(10)
                    ->get();
                    
                // Log para debug - resultado da consulta
                Log::info('Consulta de clientes realizada com sucesso', [
                    'count' => $clientes->count()
                ]);
                
                return response()->json($clientes, 200);
            } catch (\Illuminate\Database\QueryException $e) {
                // Erro específico de consulta ao banco de dados
                Log::error('Erro de consulta ao banco de dados: ' . $e->getMessage(), [
                    'sql' => $e->getSql() ?? 'N/A',
                    'bindings' => $e->getBindings() ?? [],
                    'code' => $e->getCode()
                ]);
                
                return response()->json([
                    'error' => 'Erro na consulta ao banco de dados'
                ], 500);
            }
        } catch (\Exception $e) {
            // Log detalhado do erro para depuração no servidor
            Log::error('Erro ao buscar clientes: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Retornar mensagem amigável para o usuário
            return response()->json([
                'error' => 'Ocorreu um erro ao buscar os clientes. Tente novamente.'
            ], 500);
        }
    }
}
