<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\PlanosExpiracaoNotificacao;
use App\Services\NotificacaoService;
use App\Models\PlanoEmpresa;
use Carbon\Carbon;

class NotificarPlanosExpiracao extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notificacoes:planos-expiracao';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Envia notificação sobre planos próximos de expirar nos próximos 7 dias';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Buscando planos próximos da expiração...');
        
        // Define o período de 7 dias à frente
        $dataInicio = Carbon::now()->addDay();
        $dataFim = Carbon::now()->addDays(7);
        
        // Buscar planos que expiram neste período
        $planos = PlanoEmpresa::whereBetween('data_expiracao', [$dataInicio, $dataFim])
            ->with(['empresa', 'plano'])
            ->get();
        
        if ($planos->count() == 0) {
            $this->info('Nenhum plano próximo da expiração encontrado.');
            return 0;
        }
        
        $this->info('Encontrados ' . $planos->count() . ' planos próximos da expiração.');
        
        // Agrupa planos por data de expiração
        $planosAgrupados = $planos->groupBy(function ($item) {
            return Carbon::parse($item->data_expiracao)->format('Y-m-d');
        });
        
        // Obtém lista de destinatários
        $destinatarios = NotificacaoService::getDestinatarios();
        
        try {
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new PlanosExpiracaoNotificacao($planosAgrupados));
                $this->info("Email enviado com sucesso para {$dest}");
                Log::info("Notificação de planos próximos da expiração enviada com sucesso para {$dest}");
            }
            
            $this->info('Notificação de planos próximos da expiração enviada com sucesso!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Erro ao enviar notificação de planos próximos da expiração: ' . $e->getMessage());
            Log::error('Erro ao enviar notificação de planos próximos da expiração', [
                'erro' => $e->getMessage()
            ]);
            return 1;
        }
    }
} 