<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CheckDatabaseConnection extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:check {--retry=3 : Number of retry attempts} {--delay=5 : Delay between retries in seconds}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verifica a conectividade com o banco de dados';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $retryAttempts = (int) $this->option('retry');
        $delay = (int) $this->option('delay');
        
        $this->info('Verificando conectividade com o banco de dados...');
        
        for ($attempt = 1; $attempt <= $retryAttempts; $attempt++) {
            $this->info("Tentativa {$attempt} de {$retryAttempts}...");
            
            try {
                // Tenta fazer uma consulta simples
                DB::connection()->getPdo();
                
                // Se chegou até aqui, a conexão está funcionando
                $this->info('✅ Conexão com banco de dados estabelecida com sucesso!');
                
                // Testa uma consulta simples
                $result = DB::select('SELECT 1 as test');
                if ($result) {
                    $this->info('✅ Consulta de teste executada com sucesso!');
                }
                
                // Mostra informações da conexão
                $this->info('📊 Informações da conexão:');
                $this->info('   - Driver: ' . DB::connection()->getDriverName());
                $this->info('   - Host: ' . config('database.connections.mysql.host'));
                $this->info('   - Database: ' . config('database.connections.mysql.database'));
                $this->info('   - Port: ' . config('database.connections.mysql.port'));
                
                return 0; // Sucesso
                
            } catch (\Exception $e) {
                $this->error("❌ Falha na tentativa {$attempt}: " . $e->getMessage());
                
                // Log do erro
                Log::error('Comando db:check falhou na tentativa ' . $attempt, [
                    'error' => $e->getMessage(),
                    'attempt' => $attempt,
                    'max_attempts' => $retryAttempts
                ]);
                
                if ($attempt < $retryAttempts) {
                    $this->info("⏳ Aguardando {$delay} segundos antes da próxima tentativa...");
                    sleep($delay);
                }
            }
        }
        
        $this->error('❌ Todas as tentativas falharam. Verifique a configuração do banco de dados.');
        
        // Sugestões de solução
        $this->warn('💡 Sugestões para resolver o problema:');
        $this->line('   1. Verifique se o servidor MySQL está rodando');
        $this->line('   2. Confirme as credenciais no arquivo .env');
        $this->line('   3. Verifique se o banco de dados existe');
        $this->line('   4. Confirme se o usuário tem permissões adequadas');
        $this->line('   5. Verifique se a porta está correta e acessível');
        
        return 1; // Falha
    }
} 